package com.crazyxk.imagetextbutton;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Color;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.GradientDrawable;
import android.graphics.drawable.StateListDrawable;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;

/**
 * 带图标和文字的按钮
 *
 * @author Feng Chen
 */
public class ImageTextButton extends RelativeLayout {

    public static final int ICON_POSITION_LEFT = 0;
    public static final int ICON_POSITION_TOP = 1;
    public static final int ICON_POSITION_RIGHT = 2;
    public static final int ICON_POSITION_BOTTOM = 3;

    private ImageView mIcon;
    private TextView mTxt;

    private LinearLayout mLayout;

    private int mIconPosition = ICON_POSITION_LEFT;

    public ImageTextButton(Context context) {
        super(context);
        init(context, null);
    }

    public ImageTextButton(Context context, AttributeSet attrs) {
        super(context, attrs);
        init(context, attrs);
    }

    public ImageTextButton(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init(context, attrs);
    }

    // SECTION : INNER HELPER

    private void init(Context context, AttributeSet atts) {
        if (atts == null) {
            return;
        }

        TypedArray typedArray = context.obtainStyledAttributes(atts, R.styleable.ImageTextButton);
        mIconPosition = typedArray.getInt(R.styleable.ImageTextButton_itb_icon_position, ICON_POSITION_LEFT);
        initLayout(context);

        // 填充
        initPadding(typedArray);

        // 背景
        initBackground(typedArray);

        // 文本
        initText(typedArray);

        // 图标
        initIcon(typedArray);

        typedArray.recycle();

    }


    // SECTION : INNER HELPER

    private void initLayout(Context context) {
        LayoutInflater inflater = (LayoutInflater) context.getSystemService(Context.LAYOUT_INFLATER_SERVICE);
        View view = null;
        switch (mIconPosition) {
            case ICON_POSITION_LEFT:
                view = inflater.inflate(R.layout.layout_image_text_button_left, this);
                break;
            case ICON_POSITION_TOP:
                view = inflater.inflate(R.layout.layout_image_text_button_top, this);
                break;
            case ICON_POSITION_RIGHT:
                view = inflater.inflate(R.layout.layout_image_text_button_right, this);
                break;
            case ICON_POSITION_BOTTOM:
                view = inflater.inflate(R.layout.layout_image_text_button_bottom, this);
                break;
            default:
                view = inflater.inflate(R.layout.layout_image_text_button_left, this);
                break;
        }
        mLayout = view.findViewById(R.id.area);
        mIcon = view.findViewById(R.id.iv_icon);
        mTxt = view.findViewById(R.id.tv_txt);
        mLayout.setGravity(CENTER_IN_PARENT);
    }

    private void initPadding(TypedArray typedArray) {
        int padding = typedArray.getDimensionPixelOffset(
                R.styleable.ImageTextButton_itb_padding,
                getResources().getDimensionPixelOffset(R.dimen.default_padding)
        );

        if (padding > 0) {
            setPadding(padding, padding, padding, padding);
        }
    }

    private void initBackground(TypedArray typedArray) {
        int radius = typedArray.getDimensionPixelOffset(
                R.styleable.ImageTextButton_itb_radius,
                getResources().getDimensionPixelOffset(R.dimen.default_radius)
        );

        int btnNormalBgColor = typedArray.getColor(
                R.styleable.ImageTextButton_itb_bg,
                getResources().getColor(R.color.default_bg, null)
        );

        int btnPressedBgColor = typedArray.getColor(
                R.styleable.ImageTextButton_itb_bg_pressed,
                -1
        );

        if (btnPressedBgColor == -1) {
            int alpha = Color.alpha(btnNormalBgColor);
            int red = Color.red(btnNormalBgColor);
            int green = Color.green(btnNormalBgColor);
            int blue = Color.blue(btnNormalBgColor);
            if (alpha < 0xFF) {
                btnPressedBgColor = Color.argb(0xFF, red, green, blue);
            } else {
                btnPressedBgColor = Color.argb(0x90, red, green, blue);
            }
        }
        int btnDisabledBgColor = typedArray.getColor(
                R.styleable.ImageTextButton_itb_bg_disabled,
                getResources().getColor(R.color.default_bg_disabled, null)
        );

        GradientDrawable normalShape = new GradientDrawable();
        normalShape.setColor(btnNormalBgColor);
        if (radius > 0) {
            normalShape.setCornerRadius(radius);
        }

        GradientDrawable pressedShape = new GradientDrawable();
        pressedShape.setColor(btnPressedBgColor);
        if (radius > 0) {
            pressedShape.setCornerRadius(radius);
        }

        StateListDrawable bg = new StateListDrawable();
        bg.addState(new int[]{android.R.attr.state_pressed}, pressedShape);
        bg.addState(new int[]{android.R.attr.state_enabled}, normalShape);
        setBackground(bg);
        setClickable(true);

    }

    private void initText(TypedArray typedArray) {
        // 文本
        String itbTxt = typedArray.getString(R.styleable.ImageTextButton_itb_text);
        int itbTxtSize = typedArray.getDimensionPixelSize(
                R.styleable.ImageTextButton_itb_text_size,
                getResources().getDimensionPixelSize(R.dimen.default_text_size)
        );

        if (TextUtils.isEmpty(itbTxt)) {
            mTxt.setVisibility(GONE);
        } else {
            int btnTxtColor = typedArray.getColor(
                    R.styleable.ImageTextButton_itb_text_color,
                    getResources().getColor(R.color.default_text_color, null)
            );
            mTxt.setTextColor(btnTxtColor);
            mTxt.setTextSize(px2dip(itbTxtSize));
            int margin = typedArray.getDimensionPixelOffset(
                    R.styleable.ImageTextButton_itb_icon_text_margin,
                    getResources().getDimensionPixelSize(R.dimen.default_text_margin)
            );
            ViewGroup.MarginLayoutParams params = (MarginLayoutParams) mTxt.getLayoutParams();
            switch (mIconPosition) {
                case ICON_POSITION_LEFT:
                    params.setMargins(margin, 0, 0, 0);
                    break;
                case ICON_POSITION_TOP:
                    params.setMargins(0, margin, 0, 0);
                    break;
                case ICON_POSITION_BOTTOM:
                    params.setMargins(0, 0, 0, margin);
                    break;
                case ICON_POSITION_RIGHT:
                    params.setMargins(0, 0, margin, 0);
                    break;
                default:
                    params.setMargins(margin, 0, 0, 0);
                    break;
            }

            mTxt.setLayoutParams(params);
            mTxt.setText(itbTxt);
        }
    }

    private void initIcon(TypedArray typedArray) {
        Drawable icon = typedArray.getDrawable(R.styleable.ImageTextButton_itb_icon);
        if (icon != null) {
            int iconSize = typedArray.getDimensionPixelSize(
                    R.styleable.ImageTextButton_itb_icon_size,
                    getResources().getDimensionPixelOffset(R.dimen.default_icon_width)
            );

            ViewGroup.LayoutParams params = mIcon.getLayoutParams();
            params.width = iconSize;
            params.height = iconSize;
            mIcon.setLayoutParams(params);
            mIcon.setImageDrawable(icon);

        } else {
            mIcon.setVisibility(GONE);
        }
    }

    /**
     * 根据手机的分辨率从 px(像素) 的单位 转成为 dp
     */
    private int px2dip(float pxValue) {
        final float scale = getResources().getDisplayMetrics().density;
        return (int) (pxValue / scale + 0.5f);
    }

    public ImageView getIcon() {
        return mIcon;
    }

    public TextView getText() {
        return mTxt;
    }

    public void setIcon(Drawable drawable) {
        if (mIcon != null) {
            mIcon.setImageDrawable(drawable);
        }
    }

    public void setText(String text) {
        if (mTxt != null) {
            mTxt.setText(text);
        }
    }

    public void changeBackground(Drawable drawable) {
        setBackground(drawable);
    }

}

