package com.crazyxk.pagingwindow.view;

import android.app.Activity;
import android.content.Context;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.BaseAdapter;
import android.widget.GridView;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.PopupWindow;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.viewpager.widget.PagerAdapter;
import androidx.viewpager.widget.ViewPager;

import com.crazyxk.pagingwindow.R;
import com.crazyxk.pagingwindow.bean.Page;
import com.crazyxk.pagingwindow.bean.PageItem;
import com.crazyxk.pagingwindow.bean.Paging;

import java.util.ArrayList;
import java.util.List;

/**
 * @author Feng Chen
 */
public class PagingWindow<T> {

    // 数据
    private Paging<T> mPaging;

    // 关闭按钮
    private ImageView mAction;
    private ViewPager mViewPager;
    private CircleIndicator mIndicator;
    private GridView mGridView;

    private Context mContext;

    private OnItemClickListener<T> mOnItemClickListener;
    private PopupWindow mWindow;
    private TitleGetter<T> mTitleGetter;

    private int mWidth;
    private int mHeight;

    private TextView last = null;

    public void setTitleGetter(TitleGetter<T> titleGetter) {
        mTitleGetter = titleGetter;
    }

    public void setOnItemClickListener(OnItemClickListener<T> listener) {
        mOnItemClickListener = listener;
    }

    public interface OnItemClickListener<T> {

        void onClick(int position, T t);

    }

    public interface TitleGetter<T> {

        String get(T t);

    }


    public PagingWindow(
            Context context, int width, int height, List<T> items, int pageSize,
            TitleGetter<T> titleGetter, OnItemClickListener<T> onItemClickListener
    ) {
        mContext = context;
        mPaging = new Paging<>(items, pageSize);
        mWidth = width;
        mHeight = height;
        mTitleGetter = titleGetter;
        mOnItemClickListener = onItemClickListener;
        initView();
    }

    public void show() {
        if (mWindow != null && !mWindow.isShowing()) {
            View view = ((Activity) mContext).findViewById(android.R.id.content);
            mWindow.showAtLocation(view, Gravity.BOTTOM, 0, 0);
        }
    }

    public void dismiss() {
        if (mWindow != null && mWindow.isShowing()) {
            mWindow.dismiss();
        }
    }

    private void initView() {
        View view = LayoutInflater.from(mContext).inflate(R.layout.layout_paging_window, null);
        mAction = view.findViewById(R.id.paging_window_action);
        mViewPager = view.findViewById(R.id.paging_window_viewpager);
        mIndicator = view.findViewById(R.id.paging_window_indicator);

        // 初始化 ViewPager
        initViewPager();

        mWindow = new PopupWindow(view, mWidth, mHeight);
        mWindow.setAnimationStyle(R.style.PagingWindow_Anim);
        mWindow.setBackgroundDrawable(mContext.getResources().getDrawable(R.drawable.paging_window_bg, null));

        mAction.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                dismiss();
            }
        });

    }

    private void initViewPager() {
        List<View> views = new ArrayList<>();
        for (int i = 0; i < mPaging.getPageNum(); i++) {
            View view = LayoutInflater.from(mContext).inflate(R.layout.layout_paging_window_page, null);
            mGridView = view.findViewById(R.id.paging_window_page_gridview);
            mGridView.setAdapter(new GridViewAdapter(mContext, mPaging.getPage(i)));
            views.add(view);
        }
        mViewPager.setAdapter(new ViewPagerAdapter(views));
        mIndicator.setViewPager(mViewPager);
    }


    private class ViewPagerAdapter extends PagerAdapter {

        public List<View> mViewList;

        public ViewPagerAdapter(List<View> viewList) {
            mViewList = viewList;
        }

        @Override
        public int getCount() {
            return mViewList.size();
        }

        @Override
        public boolean isViewFromObject(@NonNull View view, @NonNull Object object) {
            return view == object;
        }

        @NonNull
        @Override
        public Object instantiateItem(@NonNull ViewGroup container, int position) {
            View view = mViewList.get(position);
            container.addView(view);
            return view;
        }

        @Override
        public void destroyItem(@NonNull ViewGroup container, int position, @NonNull Object object) {
            container.removeView((View) object);
        }

    }

    public static class Builder<T> {

        private int mWidth;
        private int mHeight;

        private Context mContext;
        private TitleGetter mTitleGetter;
        private OnItemClickListener<T> mOnItemClickListener;
        private List<T> mItems;
        private int mPageSize;

        public Builder(Context context) {
            mContext = context;
        }

        public Builder setWidth(int width) {
            mWidth = width;
            return this;
        }

        public Builder setHeight(int height) {
            mHeight = height;
            return this;
        }

        public Builder setItems(List<T> items) {
            mItems = items;
            return this;
        }

        public Builder setPageSize(int pageSize) {
            mPageSize = pageSize;
            return this;
        }

        public Builder setTitleGetter(TitleGetter<T> titleGetter) {
            mTitleGetter = titleGetter;
            return this;
        }

        public Builder setOnItemClickListener(OnItemClickListener<T> listener) {
            mOnItemClickListener = listener;
            return this;
        }

        public PagingWindow<T> build() {
            return new PagingWindow<>(
                    mContext, mWidth, mHeight, mItems, mPageSize,
                    mTitleGetter, mOnItemClickListener
            );
        }

    }

    private class GridViewAdapter extends BaseAdapter {

        private Context mContext;
        private Page mPage;

        public GridViewAdapter(Context context, Page page) {
            mContext = context;
            mPage = page;
        }

        @Override
        public int getCount() {
            return mPage.getItems().size();
        }

        @Override
        public Object getItem(int position) {
            return mPage.getItems().get(position);
        }

        @Override
        public long getItemId(int position) {
            return position;
        }

        @Override
        public View getView(final int position, View convertView, ViewGroup parent) {
            ViewHolder viewHolder;
            final PageItem pageItem = mPage.getItems().get(position);
            if (convertView == null) {
                convertView = LayoutInflater.from(mContext).inflate(R.layout.layout_paging_window_page_item, null);
                viewHolder = new ViewHolder();
                viewHolder.tvNo = convertView.findViewById(R.id.paging_window_page_item_no);
                viewHolder.tvTitle = convertView.findViewById(R.id.paging_window_page_item_title);

                viewHolder.tvTitle.setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {

                        if (last != null) {
                            last.setTextColor(mContext.getColor(R.color.gray));
                        }

                        ((TextView) v).setTextColor(mContext.getColor(R.color.yellow));
                        if (mOnItemClickListener != null) {
                            mOnItemClickListener.onClick(pageItem.getIdx(), (T) pageItem.getItem());
                        }

                        last = (TextView) v;
                    }
                });

                convertView.setTag(viewHolder);

            } else {
                viewHolder = (ViewHolder) convertView.getTag();
            }
            fillValue(pageItem, viewHolder);
            return convertView;
        }

        private void fillValue(PageItem pageItem, ViewHolder viewHolder) {
            viewHolder.tvNo.setText(pageItem.getId() + ".");
            if (mTitleGetter != null) {
                viewHolder.tvTitle.setText(mTitleGetter.get((T) pageItem.getItem()));
            }
        }
    }

    private class ViewHolder {
        TextView tvNo;
        TextView tvTitle;
    }


}
