package com.cssw.afr.core;

import com.cssw.afr.AfrNet;
import com.cssw.afr.Rect;
import com.cssw.afr.core.dao.AfrDao;
import com.cssw.afr.core.exception.*;
import com.cssw.afr.core.model.dto.Candidate;
import com.cssw.afr.core.model.dto.FaceRect;
import com.cssw.afr.core.model.dto.SearchResult;
import com.cssw.afr.core.model.entity.PersonDO;
import com.cssw.afr.core.model.req.*;
import com.cssw.afr.core.model.resp.DetectResp;
import com.cssw.afr.core.model.resp.SearchResp;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.util.ArrayList;
import java.util.List;

/**
 * @author Feng Chen
 */
@Slf4j
@RequiredArgsConstructor
public class AfrApiImpl implements AfrApi {

    private final AfrDao afrDao;

    @Override
    public DetectResp detect(DetectReq req) {
        List<Rect> list = AfrNet.detect(req.getFilePath());
        List<FaceRect> rects = new ArrayList<>();
        for (Rect li : list) {
            FaceRect rect = new FaceRect();
            rect.setX(li.getX());
            rect.setY(li.getY());
            rect.setWidth(li.getWidth());
            rect.setHeight(li.getHeight());
            rects.add(rect);
        }
        return DetectResp.builder()
                .rects(rects)
                .build();
    }

    @Override
    public void createGroup(GroupCreateReq req) {
        checkGroupAlreadyExists(req.getName());
        afrDao.createGroup(req.getName());
    }

    @Override
    public void deleteGroup(GroupDeleteReq req) {
        checkGroupExists(req.getName());
        afrDao.deleteGroup(req.getName());
    }

    @Override
    public void createPerson(PersonCreateReq req) {
        checkGroupExists(req.getGroupName());
        checkPersonAlreadyExists(req.getGroupName(), req.getId());
        List<Rect> rects = AfrNet.detect(req.getFilePath());
        checkNoneFaces(rects);
        checkMultiFaces(rects);
        float[] features = AfrNet.extract(req.getFilePath(), rects.get(0));
        afrDao.createPerson(req.getGroupName(), req.getId(), features);
    }

    @Override
    public void updatePerson(PersonUpdateReq req) {
        checkGroupExists(req.getGroupName());
        checkPersonExists(req.getGroupName(), req.getId());
        List<Rect> rects = AfrNet.detect(req.getFilePath());
        checkNoneFaces(rects);
        checkMultiFaces(rects);
        float[] features = AfrNet.extract(req.getFilePath(), rects.get(0));
        afrDao.updatePerson(req.getGroupName(), req.getId(), features);
    }

    @Override
    public void deletePerson(PersonDeleteReq req) {
        checkGroupExists(req.getGroupName());
        checkPersonExists(req.getGroupName(), req.getId());
        afrDao.deletePerson(req.getGroupName(), req.getId());
    }

    @Override
    public SearchResp search(SearchReq req) {
        checkGroupExists(req.getGroupName());
        List<Rect> rects = AfrNet.detect(req.getFilePath());
        checkNoneFaces(rects);
        List<SearchResult> results = new ArrayList<>();
        for (Rect rect : rects) {
            float[] features = AfrNet.extract(req.getFilePath(), rect);
            List<PersonDO> list = afrDao.searchPerson(req.getGroupName(), features);
            List<Candidate> candidates = new ArrayList<>();

            for (PersonDO li : list) {
                Candidate candidate = new Candidate();
                candidate.setId(li.getId());
                candidate.setScore(li.getScore());
                candidates.add(candidate);
            }

            FaceRect faceRect = new FaceRect();
            faceRect.setX(rect.getX());
            faceRect.setY(rect.getY());
            faceRect.setWidth(rect.getWidth());
            faceRect.setHeight(rect.getHeight());

            results.add(
                    SearchResult.builder()
                            .candidates(candidates)
                            .rect(faceRect)
                            .build()
            );
        }
        return SearchResp.builder()
                .results(results)
                .build();
    }

    private void checkGroupAlreadyExists(String groupName) {
        if (afrDao.containsGroup(groupName)) {
            throw new GroupAlreadyExistsException();
        }
    }

    private void checkGroupExists(String groupName) {
        if (!afrDao.containsGroup(groupName)) {
            throw new GroupNotExistsException();
        }
    }

    private void checkPersonAlreadyExists(String groupName, String id) {
        if (afrDao.containsPerson(groupName, id)) {
            throw new PersonAlreadyExistsException();
        }
    }

    private void checkPersonExists(String groupName, String id) {
        if (!afrDao.containsPerson(groupName, id)) {
            throw new PersonNotExistsException();
        }
    }

    private void checkMultiFaces(List<Rect> rects) {
        if (rects.size() > 1) {
            throw new MultiFacesDetectedException();
        }
    }

    private void checkNoneFaces(List<Rect> rects) {
        if (rects.isEmpty()) {
            throw new NoneFacesDetectedException();
        }
    }

}
