package com.cssw.afr.core.dao;

import com.cssw.afr.autoconfigure.AfrProperties;
import com.cssw.afr.core.model.entity.FaceDO;
import com.cssw.afr.core.model.entity.PersonDO;
import io.milvus.v2.client.ConnectConfig;
import io.milvus.v2.client.MilvusClientV2;
import io.milvus.v2.common.DataType;
import io.milvus.v2.common.IndexParam;
import io.milvus.v2.service.collection.request.AddFieldReq;
import io.milvus.v2.service.collection.request.CreateCollectionReq;
import io.milvus.v2.service.collection.request.DropCollectionReq;
import io.milvus.v2.service.collection.request.HasCollectionReq;
import io.milvus.v2.service.database.request.CreateDatabaseReq;
import io.milvus.v2.service.database.response.ListDatabasesResp;
import io.milvus.v2.service.vector.request.DeleteReq;
import io.milvus.v2.service.vector.request.GetReq;
import io.milvus.v2.service.vector.request.InsertReq;
import io.milvus.v2.service.vector.request.SearchReq;
import io.milvus.v2.service.vector.request.data.FloatVec;
import io.milvus.v2.service.vector.response.GetResp;
import io.milvus.v2.service.vector.response.SearchResp;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.InitializingBean;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import static com.cssw.afr.core.util.GsonUtil.toJsonObjectList;
import static io.milvus.v2.client.MilvusClientV2.CreateSchema;

/**
 * @author Feng Chen
 */
@Slf4j
public class AfrDaoImpl implements AfrDao, InitializingBean {

    private final AfrProperties properties;
    private final MilvusClientV2 milvus;

    public AfrDaoImpl(AfrProperties properties) {
        this.properties = properties;
        milvus = new MilvusClientV2(
                ConnectConfig.builder()
                        .uri(properties.getUrl())
                        .username(properties.getUsername())
                        .password(properties.getPassword())
                        .build()
        );
    }

    @Override
    public void createGroup(String groupName) {
        milvus.createCollection(
                CreateCollectionReq.builder()
                        .databaseName(properties.getDbName())
                        .collectionName(groupName)
                        .collectionSchema(
                                CreateSchema()
                                        .addField(
                                                AddFieldReq.builder()
                                                        .fieldName("id")
                                                        .dataType(DataType.VarChar)
                                                        .maxLength(50)
                                                        .isPrimaryKey(true)
                                                        .build()
                                        )
                                        .addField(
                                                AddFieldReq.builder()
                                                        .fieldName("features")
                                                        .dataType(DataType.FloatVector)
                                                        .dimension(1024)
                                                        .build()
                                        )
                        )
                        .indexParam(
                                IndexParam.builder()
                                        .fieldName("features")
                                        .indexType(IndexParam.IndexType.AUTOINDEX)
                                        .metricType(IndexParam.MetricType.COSINE)
                                        .build()
                        )
                        .build()
        );
    }

    @Override
    public void deleteGroup(String groupName) {
        milvus.dropCollection(
                DropCollectionReq.builder()
                        .databaseName(properties.getDbName())
                        .collectionName(groupName)
                        .build()
        );
    }

    @Override
    public Boolean containsGroup(String groupName) {
        Boolean exists = milvus.hasCollection(
                HasCollectionReq.builder()
                        .databaseName(properties.getDbName())
                        .collectionName(groupName)
                        .build()
        );
        return exists;
    }

    @Override
    public void createPerson(String group, String id, float[] features) {
        milvus.insert(
                InsertReq.builder()
                        .databaseName(properties.getDbName())
                        .collectionName(group)
                        .data(
                                toJsonObjectList(
                                        FaceDO.builder()
                                                .id(id)
                                                .features(features)
                                                .build()
                                )
                        )
                        .build()
        );
    }

    @Override
    public void updatePerson(String group, String id, float[] features) {
        deletePerson(group, id);
        createPerson(group, id, features);
    }

    @Override
    public void deletePerson(String group, String id) {
        milvus.delete(
                DeleteReq.builder()
                        .databaseName(properties.getDbName())
                        .collectionName(group)
                        .ids(Collections.singletonList(id))
                        .build()
        );
    }

    @Override
    public Boolean containsPerson(String group, String id) {
        GetResp resp = milvus.get(
                GetReq.builder()
                        .databaseName(properties.getDbName())
                        .collectionName(group)
                        .ids(Collections.singletonList(id))
                        .build()
        );
        return !resp.getGetResults().isEmpty();
    }

    @Override
    public List<PersonDO> searchPerson(String group, float[] features) {
        SearchResp resp = milvus.search(
                SearchReq.builder()
                        .databaseName(properties.getDbName())
                        .collectionName(group)
                        .data(Collections.singletonList(new FloatVec(features)))
                        .annsField("features")
                        .topK(3)
                        .build()
        );
        List<PersonDO> list = new ArrayList<>();
        List<SearchResp.SearchResult> searchResults = resp.getSearchResults().get(0);
        if (searchResults != null && !searchResults.isEmpty()) {
            for (SearchResp.SearchResult searchResult : searchResults) {
                list.add(
                        PersonDO.builder()
                                .id(String.valueOf(searchResult.getId()))
                                .score(searchResult.getScore())
                                .build()
                );
            }
        }
        return list;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        try {
            ListDatabasesResp listDatabasesResp = milvus.listDatabases();
            if (!listDatabasesResp.getDatabaseNames().contains(properties.getDbName())) {
                milvus.createDatabase(
                        CreateDatabaseReq.builder()
                                .databaseName(properties.getDbName())
                                .build()
                );
            }
        } catch (Exception e) {
            log.error("", e);
        }
    }

}
