package com.cssw.iotnet;

import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.eclipse.paho.client.mqttv3.*;
import org.eclipse.paho.client.mqttv3.persist.MemoryPersistence;

import java.nio.charset.StandardCharsets;

import static com.cssw.iotnet.IotnetConsts.IOTNET_TOPIC_DEFAULT_PUB;
import static com.cssw.iotnet.IotnetConsts.IOTNET_TOPIC_DEFAULT_SUB;

/**
 * @author Feng Chen
 */
@Slf4j
public class IotnetClient {

    private final IotnetClientOptions iotnetClientOptions;

    @Setter
    private IotnetMessageListener iotnetMessageListener;

    private MqttClient mqttClient;

    public IotnetClient(IotnetClientOptions iotnetClientOptions) {
        this.iotnetClientOptions = iotnetClientOptions;
        newMqttClient();
    }

    /**
     * 连接
     */
    public void connect() {
        try {
            // 连接
            mqttClient.connect(newMqttConnectOptions());

            // 订阅默认订阅主题
            subscribe(IOTNET_TOPIC_DEFAULT_SUB);

        } catch (MqttException e) {
            log.error("", e);
            throw new IotnetException(e);
        }
    }

    /**
     * 断开连接
     */
    public void disconnect() {
        try {
            // 取消订阅默认订阅主题
            unsubscribe(IOTNET_TOPIC_DEFAULT_SUB);

            // 断开连接
            mqttClient.disconnect();

        } catch (MqttException e) {
            log.error("", e);
            throw new IotnetException(e);
        }
    }

    /**
     * 给默认主题发送消息
     *
     * @param message 待发送消息
     */
    public void publish(String message) {
        publish(IOTNET_TOPIC_DEFAULT_PUB, message);
    }

    /**
     * 给指定主题发布消息
     *
     * @param topic   主题
     * @param message 消息
     */
    public void publish(String topic, String message) {
        try {
            mqttClient.publish(topic, new MqttMessage(message.getBytes(StandardCharsets.UTF_8)));
        } catch (MqttException e) {
            log.error("", e);
            throw new IotnetException(e);
        }
    }

    /**
     * 订阅主题
     *
     * @param topic 待订阅主题
     */
    public void subscribe(String topic) {
        try {
            mqttClient.subscribe(topic, new IMqttMessageListener() {
                @Override
                public void messageArrived(String topic, MqttMessage message) throws Exception {
                    if (iotnetMessageListener != null) {
                        iotnetMessageListener.onMessageArrived(
                                IotnetMessage.builder()
                                        .topic(topic)
                                        .messageId(message.getId())
                                        .payload(message.getPayload())
                                        .build()
                        );
                    }
                }
            });
        } catch (MqttException e) {
            log.error("", e);
            throw new IotnetException(e);
        }
    }

    /**
     * 取消主题订阅
     *
     * @param topic 待取消订阅主题
     */
    public void unsubscribe(String topic) {
        try {
            mqttClient.unsubscribe(topic);
        } catch (MqttException e) {
            log.error("", e);
            throw new IotnetException(e);
        }
    }

    // 创建 Mqtt 客户端
    private void newMqttClient() {
        try {
            mqttClient = new MqttClient(
                    iotnetClientOptions.getServerUrl(),
                    iotnetClientOptions.getDeviceName(),
                    new MemoryPersistence()
            );
        } catch (MqttException e) {
            log.error("", e);
            throw new IotnetException(e);
        }
    }

    private MqttConnectOptions newMqttConnectOptions() {
        MqttConnectOptions mqttConnectOptions = new MqttConnectOptions();
        mqttConnectOptions.setCleanSession(true);
        mqttConnectOptions.setAutomaticReconnect(true);
        mqttConnectOptions.setConnectionTimeout(iotnetClientOptions.getConnectTimeout());
        mqttConnectOptions.setKeepAliveInterval(iotnetClientOptions.getKeepAlive());
        mqttConnectOptions.setMaxReconnectDelay(iotnetClientOptions.getReconnectInterval() * 1000);
        mqttConnectOptions.setUserName(iotnetClientOptions.getProductKey() + "#" + iotnetClientOptions.getDeviceName());
        mqttConnectOptions.setPassword(iotnetClientOptions.getDeviceSecret().toCharArray());
        return mqttConnectOptions;
    }

}
