package com.cssw.swagger;

import io.swagger.v3.oas.models.Components;
import io.swagger.v3.oas.models.OpenAPI;
import io.swagger.v3.oas.models.info.Contact;
import io.swagger.v3.oas.models.info.Info;
import io.swagger.v3.oas.models.security.*;
import lombok.Setter;
import org.springframework.beans.factory.FactoryBean;

import java.util.Collections;

import static com.cssw.swagger.SwaggerProperties.GrantTypes.*;

/**
 * @author Feng Chen
 */
@Setter
public class OpenApiFactoryBean implements FactoryBean<OpenAPI> {

    private SwaggerProperties swaggerProperties;

    @Override
    public OpenAPI getObject() throws Exception {
        return newOpenAPI();
    }

    @Override
    public Class<?> getObjectType() {
        return OpenAPI.class;
    }

    private OpenAPI newOpenAPI() {
        OpenAPI openAPI = new OpenAPI();
        openAPI.info(newInfo());
        if (swaggerProperties.getAuthorization().isEnabled()) {
            openAPI.components(newApikeyComponents());
            openAPI.security(Collections.singletonList(
                    new SecurityRequirement()
                            .addList(swaggerProperties.getAuthorization().getName()))
            );
        } else if (swaggerProperties.getOauth2().isEnabled()) {
            openAPI.components(newOauth2Components());
            openAPI.security(Collections.singletonList(
                    new SecurityRequirement()
                            .addList(swaggerProperties.getOauth2().getName())
            ));
        }
        return openAPI;
    }

    private Info newInfo() {
        return new Info()
                .title(swaggerProperties.getTitle())
                .description(swaggerProperties.getDescription())
                .termsOfService(swaggerProperties.getTermsOfServiceUrl())
                .contact(
                        new Contact()
                                .name(swaggerProperties.getContact().getName())
                                .email(swaggerProperties.getContact().getEmail())
                                .url(swaggerProperties.getContact().getUrl()))
                .version(swaggerProperties.getVersion())
                ;
    }

    // apikey
    private Components newApikeyComponents() {
        SwaggerProperties.Authorization authorization = swaggerProperties.getAuthorization();
        Components components = new Components();
        SecurityScheme securityScheme = new SecurityScheme();
        securityScheme.type(SecurityScheme.Type.APIKEY);
        securityScheme.in(SecurityScheme.In.HEADER);
        securityScheme.name(authorization.getKeyName());
        components.addSecuritySchemes(authorization.getName(), securityScheme);
        return components;
    }

    // oauth2
    private Components newOauth2Components() {
        SwaggerProperties.Oauth2 oauth2 = swaggerProperties.getOauth2();
        Components components = new Components();
        SecurityScheme securityScheme = new SecurityScheme();
        securityScheme.type(SecurityScheme.Type.OAUTH2);
        OAuthFlows oAuthFlows = new OAuthFlows();
        if (oauth2.getGrantType() == IMPLICIT) {
            oAuthFlows.implicit(
                    new OAuthFlow()
                            .authorizationUrl(oauth2.getAuthorizeUrl())
                            .scopes(newScopes(oauth2))
            );
        } else if (oauth2.getGrantType() == CLIENT_CREDENTIALS) {
            oAuthFlows.clientCredentials(
                    new OAuthFlow()
                            .tokenUrl(oauth2.getTokenUrl())
                            .scopes(newScopes(oauth2))
            );
        } else if (oauth2.getGrantType() == AUTHORIZATION_CODE) {
            oAuthFlows.authorizationCode(
                    new OAuthFlow()
                            .authorizationUrl(oauth2.getAuthorizeUrl())
                            .tokenUrl(oauth2.getTokenUrl())
                            .scopes(newScopes(oauth2))
            );
        } else if (oauth2.getGrantType() == PASSWORD) {
            oAuthFlows.password(
                    new OAuthFlow()
                            .tokenUrl(oauth2.getTokenUrl())
                            .scopes(newScopes(oauth2))
            );
        }
        securityScheme.setFlows(oAuthFlows);
        components.addSecuritySchemes(oauth2.getName(), securityScheme);
        return components;
    }

    private Scopes newScopes(SwaggerProperties.Oauth2 oauth2) {
        Scopes scopes = new Scopes();
        for (SwaggerProperties.AuthorizationScope scope : oauth2.getScopes()) {
            scopes.addString(scope.getScope(), scope.getDescription());
        }
        return scopes;
    }

}
