package com.ydn.map;

import android.app.ActivityManager;
import android.content.Context;
import android.content.pm.ConfigurationInfo;
import android.opengl.GLSurfaceView;
import android.util.AttributeSet;
import android.view.MotionEvent;
import android.view.View;

import androidx.annotation.Nullable;

import static android.view.MotionEvent.ACTION_DOWN;
import static android.view.MotionEvent.ACTION_UP;

/**
 * @author Feng Chen
 */
public class MapView extends GLSurfaceView {

    private MapRender mMapRender;
    private Context mContext;

    private final int CONTEXT_CLIENT_VERSION = 3;

    private float sx = 0.0f, sy = 0.0f;
    private float ex = 0.0f, ey = 0.0f;
    private double len;

    // 点击事件是否完成
    private boolean isOver = false;

    private static final int MAX_DNL = 10;

    public MapRender getMapRender() {
        return mMapRender;
    }

    public void setMapRender(MapRender mapRender) {
        mMapRender = mapRender;
        initView();
    }

    public MapView(Context context) {
        this(context, null);
    }

    public MapView(Context context, @Nullable AttributeSet attrs) {
        super(context, attrs);
        mContext = context;
    }

    public void locate(float x, float y, float angle) {
        if (mMapRender == null) {
            return;
        }
        mMapRender.locate(x, y, angle);
        requestRender();
    }

    public void mark(String ids) {
        if (mMapRender == null) {
            return;
        }
        mMapRender.mark(ids);
        requestRender();
    }

    public void floor(int floor) {
        if (mMapRender == null) {
            return;
        }
        mMapRender.floor(floor);
        requestRender();
    }

    // 缩放操作
    public void zoom(float scale) {
        mMapRender.zoom(scale);
        requestRender();
    }

    // 移动操作
    public void move(float x, float y) {
        mMapRender.move(x, y);
        requestRender();
    }

    // 点击操作
    public void click(float x, float y) {
        mMapRender.click(x, y);
        requestRender();
    }

    public void nav(String id) {
        mMapRender.nav(id);
        requestRender();
    }

    // SECTION : INNER HELPER

    private void initView() {
        setupOpenGLES();
        initTouchEvent();
    }

    private void setupOpenGLES() {
        if (detachOpenGLES30()) {
            setEGLContextClientVersion(CONTEXT_CLIENT_VERSION);
            setRenderer(mMapRender);
            setRenderMode(RENDERMODE_WHEN_DIRTY);
        }
    }

    private void initTouchEvent() {
        /**
         * 一根手指 拖动
         * ACTION_DOWN -> ACTION_MOVE（N） -> ACTION_UP
         * 0					2					1
         *
         * 一根手指 点击
         * ACTION_DOWN -> ACTION_MOVE（N） -> ACTION_UP
         * 0					2					1
         */
        setOnTouchListener(new OnTouchListener() {
            @Override
            public boolean onTouch(View v, MotionEvent event) {
                if (event.getActionMasked() == ACTION_DOWN) {
                    sx = event.getX();
                    sy = event.getY();
                } else if (event.getActionMasked() == ACTION_UP) {
                    ex = event.getX();
                    ey = event.getY();
                    isOver = true;
                }
                if (isOver) {
                    int xLen = Math.abs((int) sx - (int) ex);
                    int yLen = Math.abs((int) sy - (int) ey);
                    len = Math.sqrt((double) xLen * xLen + (double) yLen * yLen);

                    if (len < MAX_DNL) {
                        click(sx, sy);
                    } else {
                        move(ex - sx, ey - sy);
                    }
                }
                isOver = false;
                return true;
            }
        });

    }


    // SECTION : INNER HELPER

    private boolean detachOpenGLES30() {
        ActivityManager am = (ActivityManager) mContext.getSystemService(Context.ACTIVITY_SERVICE);
        ConfigurationInfo info = am.getDeviceConfigurationInfo();
        return (info.reqGlEsVersion >= 0x30000);
    }


}
