package com.ydn.simpleserial.adapter;

import com.ydn.simpleserial.configuration.ZkSerialConfiguration;
import org.apache.curator.RetryPolicy;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.framework.recipes.locks.InterProcessLock;
import org.apache.curator.framework.recipes.locks.InterProcessMutex;
import org.apache.curator.retry.ExponentialBackoffRetry;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;

import java.util.concurrent.TimeUnit;

/**
 * @author Feng Chen
 */
public class ZkSerialAdapter implements SerialAdapter {

    private ZkSerialConfiguration configuration;
    private CuratorFramework curator;

    public ZkSerialAdapter(ZkSerialConfiguration configuration) {
        this.configuration = configuration;
        initCurator();
    }

    @Override
    public int get() throws Exception {
        return getAndIncrement(null);
    }

    @Override
    public int get(String key) throws Exception {
        return getAndIncrement(key);
    }

    @Override
    public void start() throws Exception {
        curator.start();
    }

    @Override
    public void close() throws Exception {
        if (curator != null) {
            curator.close();
        }
    }

    // SECTION : INNER HELPER

    private void initCurator() {
        RetryPolicy retryPolicy = new ExponentialBackoffRetry(configuration.getBaseSleepTimeMs(), configuration.getMaxRetries());
        this.curator = CuratorFrameworkFactory.newClient(
                configuration.getZkHosts(),
                configuration.getSessionTimeoutMs(),
                configuration.getConnectionTimeoutMs(),
                retryPolicy
        );
    }

    private int getAndIncrement(String key) throws Exception {
        int value = 0;
        InterProcessLock lock = getLock(key);
        String path = getPath(key);
        try {
            if (!lock.acquire(configuration.getMaxWait(), TimeUnit.MILLISECONDS)) {
                throw new IllegalStateException("Could not acquire the lock");
            }

            byte[] bytes;
            try {
                bytes = curator.getData().forPath(path);
            } catch (KeeperException.NoNodeException e) {
                // 不存在新增
                mkdir(path);
                bytes = curator.getData().forPath(path);
            }

            if (bytes != null) {
                value = Integer.parseInt(new String(bytes));
                if (value > configuration.getMaxValue()) {
                    value = 1;
                }
            }

            curator.setData().forPath(path, String.valueOf(value + 1).getBytes());
        } finally {
            lock.release();
        }
        return value;
    }

    private void mkdir(String path) {
        try {
            if (curator.checkExists().forPath(path) == null) {
                curator.create().creatingParentsIfNeeded().withMode(CreateMode.PERSISTENT).forPath(path, "1".getBytes());
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private InterProcessLock getLock(String key) {
        InterProcessMutex lock;
        if (key != null) {
            lock = new InterProcessMutex(curator, String.format("/simpleserial/%s/%s/lock", configuration.getNamespace(), key));
        } else {
            lock = new InterProcessMutex(curator, String.format("/simpleserial/%s/lock", configuration.getNamespace()));
        }
        return lock;
    }

    private String getPath(String key) {
        String path;
        if (key != null) {
            path = String.format("/simpleserial/%s/%s/serial", configuration.getNamespace(), key);
        } else {
            path = String.format("/simpleserial/%s/serial", configuration.getNamespace());
        }
        return path;
    }

}
