package com.ydn.web.appserver.core;

import com.ydn.web.appserver.ContextListener;
import com.ydn.web.appserver.Interceptor;
import com.ydn.web.appserver.WebAppServer;
import com.ydn.web.appserver.spring.SpringServerInjector;
import com.ydn.web.appserver.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Properties;

/**
 * @author Feng Chen
 */
public class AppServerFactory {

    private static final Logger logger = LoggerFactory.getLogger(AppServerFactory.class);

    public static final String PROPERTY_IP = "ip";

    public static final String PROPERTY_CONTEXT_PATH = "context-path";

    public static final String PROPERTY_PORT = "port";

    public static final String PROPERTY_MAX_ACTIVE_WORKERS = "max-active-workers";

    public static final String PROPERTY_CONTROLLER_PACKAGES = "controller-packages";

    public static final String PROPERTY_SPRING_CONFIG = "spring-config";

    public static final String PROPERTY_CONTEXT_LISTENER = "context-listener";

    public static final String PROPERTY_INTERCEPTORS = "interceptors";

    public static final String PROPERTY_USERNAME = "username";

    public static final String PROPERTY_PASSWORD = "password";

    public static WebAppServer createServer(Properties properties) {
        String contextPath = getProperty(properties, PROPERTY_CONTEXT_PATH, ServerConfiguration.DEFAULT_CONTEXT_PATH);
        String ip = getProperty(properties, PROPERTY_IP, ServerConfiguration.DEFAULT_LISTEN_IP);
        int port = Integer.parseInt(
                getProperty(properties, PROPERTY_PORT, String.valueOf(ServerConfiguration.DEFAULT_LISTEN_PORT))
        );

        int maxActiveWorkers = Integer.parseInt(
                getProperty(properties, PROPERTY_MAX_ACTIVE_WORKERS, String.valueOf(ServerConfiguration.DEFAULT_MAX_PROCESSORS))
        );

        WebAppServer server = new WebAppServer(contextPath, ip, port, maxActiveWorkers);

        // 其他配置
        configServer(properties, server);
        return server;
    }

    private static void configServer(Properties properties, WebAppServer server) {
        // 初始化 Spring
        String springConfigFile = properties.getProperty(PROPERTY_SPRING_CONFIG);
        if (StringUtil.isNotEmpty(springConfigFile)) {
            SpringServerInjector.inject(server, springConfigFile);
        }

        // activity package
        String packages = properties.getProperty(PROPERTY_CONTROLLER_PACKAGES);
        if (StringUtil.isNotEmpty(packages)) {
            server.setPackages(packages.split(","));
        }

        // listener
        String contextListenerClassName = properties.getProperty(PROPERTY_CONTEXT_LISTENER);
        if (StringUtil.isNotEmpty(contextListenerClassName)) {
            initListener(server, contextListenerClassName);
        }

        // interceptor
        String interceptors = properties.getProperty(PROPERTY_INTERCEPTORS);
        if (StringUtil.isNotEmpty(interceptors)) {
            initInterceptors(server, interceptors);
        }

        // 初始化 user/pwd
        String username = properties.getProperty(PROPERTY_USERNAME);
        if (StringUtil.isNotEmpty(username)) {
            server.setAdminUser(username);
        }
        String password = properties.getProperty(PROPERTY_PASSWORD);
        if (StringUtil.isNotEmpty(password)) {
            server.setAdminPwd(password);
        }

    }

    private static void initInterceptors(WebAppServer server, String interceptors) {
        for (String clazz : interceptors.split(",")) {
            if (StringUtil.isNotEmpty(clazz)) {
                Interceptor interceptor = BeanFactory.getBeanByClass(server, clazz);
                if (interceptor != null) {
                    server.addInterceptor(interceptor);
                }
            }
        }
    }

    private static void initListener(WebAppServer server, String contextListenerClassName) {
        ContextListener listener = BeanFactory.getBeanByClass(server, contextListenerClassName);
        if (listener != null) {
            server.setContextListener(listener);
        }
    }

    protected static String getProperty(Properties properties, String key, String defaultValue) {
        Object value = properties.get(key);

        if (value == null) {
            return defaultValue;
        } else {
            return value.toString();
        }
    }

}
