package com.ydn.web.appserver.core;

import com.ydn.web.appserver.util.Counter;
import com.ydn.web.appserver.util.DateUtil;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * @author Feng Chen
 */
public class ServerStat {

    // 接口 - 统计信息
    private final Map<String, ActionStat> stats = new HashMap<>();
    private final Map<String, Counter> daily = new HashMap<>();
    private final Map<String, Counter> monthly = new HashMap<>();

    private final long ctime;
    private AtomicInteger total;

    public ServerStat() {
        ctime = DateUtil.now();
        total = new AtomicInteger(0);
    }

    /**
     * 添加成功执行记录
     *
     * @param actionName
     * @param elapse
     */
    public void succ(String actionName, long elapse) {
        ActionStat stat = getStat(actionName);
        stat.succ(elapse);
        addDaily();
        addMonthly();
        total.incrementAndGet();
    }

    /**
     * 添加失败执行记录
     *
     * @param actionName
     * @param elapse
     * @param e
     */
    public void fail(String actionName, long elapse, Exception e) {
        ActionStat stat = getStat(actionName);
        stat.fail(elapse, e);
        addDaily();
        addMonthly();
        total.incrementAndGet();
    }

    /**
     * 查询所有统计信息
     *
     * @return
     */
    public List<ActionStat> getStats() {
        return new ArrayList<>(stats.values());
    }

    /**
     * 获取请求统计
     *
     * @return
     */
    public List<ReqStat> getReqStats() {
        List<ReqStat> result = new ArrayList<>();
        List<ActionStat> snapshots = new ArrayList<>(stats.values());
        snapshots.stream().forEach(a -> {
            result.add(new ReqStat(a.getName(), a.getDesc(), a.getTotalCnt()));
        });
        return result;
    }

    /**
     * 获取性能统计
     *
     * @return
     */
    public List<PrefStat> getPrefStats() {
        List<PrefStat> result = new ArrayList<>();
        List<ActionStat> snapshots = new ArrayList<>(stats.values());
        snapshots.stream().forEach(a -> {
            result.add(new PrefStat(a.getName(), a.getDesc(), a.getTotalCst() * 1.0 / a.getTotalCnt()));
        });
        return result;
    }

    public int getDaily() {
        String date = DateUtil.getDate();
        return daily.containsKey(date) ? daily.get(date).getNum() : 0;
    }

    public int getMonthly() {
        String month = DateUtil.getMonth();
        return monthly.containsKey(month) ? monthly.get(month).getNum() : 0;
    }

    public long getCtime() {
        return ctime;
    }

    public int getTotal() {
        return total.get();
    }

    // SECTION : INNER HELPER

    private ActionStat getStat(String actionName) {
        ActionStat stat = stats.get(actionName);
        if (stat == null) {
            synchronized (stats) {
                stat = stats.get(actionName);
                if (stat == null) {

                    stat = new ActionStat(actionName, getDesc(actionName));
                    stats.put(actionName, stat);
                }
            }
        }
        return stat;
    }

    private String getDesc(String actionName) {
        return AppServerCore.inst()
                .getActionMappings()
                .find(actionName)
                .getAnno()
                .desc();
    }

    private void addDaily() {
        String date = DateUtil.getDate();
        if (!daily.containsKey(date)) {
            synchronized (daily) {
                if (!daily.containsKey(date)) {
                    daily.put(date, new Counter());
                }
            }
        }
        daily.get(date).inc();
    }


    private void addMonthly() {
        String month = DateUtil.getMonth();
        if (!monthly.containsKey(month)) {
            synchronized (monthly) {
                if (!monthly.containsKey(month)) {
                    monthly.put(month, new Counter());
                }
            }
        }
        monthly.get(month).inc();
    }


}
