package com.ydn.web.appserver.spring;


import com.ydn.web.appserver.AppServerException;
import com.ydn.web.appserver.WebAppServer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;
import org.springframework.context.support.GenericXmlApplicationContext;

/**
 * @author Feng Chen
 */
public class SpringServerInjector {

    public static final String APPCONTEXT_PROPERTY_NAME = "spring-app-context";

    private static final Logger logger = LoggerFactory.getLogger(SpringServerInjector.class);

    public static void init(WebAppServer server, ApplicationContext applicationContext) {
        server.setProperty(APPCONTEXT_PROPERTY_NAME, applicationContext);
        SpringControllerFactory controllerFactory = new SpringControllerFactory();
        controllerFactory.setApplicationContext(applicationContext);
        server.setControllerFactory(controllerFactory);
    }

    public static void inject(WebAppServer server, String springConfigFile) {
        // 已被 Spring 初始化，则跳过此步
        if (isSpringInjected(server)) {
            return;
        }

        ApplicationContext context;
        if (springConfigFile.startsWith("classpath:")) {
            // 本地 classpath 加载
            context = new ClassPathXmlApplicationContext(springConfigFile);
        } else {
            // 远程加载
            context = new GenericXmlApplicationContext(springConfigFile);
        }
        init(server, context);
    }

    public static boolean isSpringInjected(WebAppServer server) {
        return server.getProperty(APPCONTEXT_PROPERTY_NAME) != null
                &&
                server.getProperty(APPCONTEXT_PROPERTY_NAME) instanceof ApplicationContext
                ;
    }

    public static <T> T getBeanByClass(WebAppServer appServer, String className) {
        ApplicationContext context = appServer.getProperty(APPCONTEXT_PROPERTY_NAME);
        try {
            if (context == null) {
                return null;
            } else {
                return (T) context.getBean(Class.forName(className));
            }
        } catch (Exception e) {
            logger.error("", e);
            throw new AppServerException(e);
        }
    }

}
