package com.ydn.web.appserver;

import com.ydn.web.appserver.core.AppServerCore;
import com.ydn.web.appserver.core.CompositeResourceManager;
import com.ydn.web.appserver.core.ControllerFactory;
import com.ydn.web.appserver.core.ServerConfiguration;
import com.ydn.web.appserver.interceptor.CorsInterceptor;
import io.undertow.Handlers;
import io.undertow.Undertow;
import io.undertow.server.HttpHandler;
import io.undertow.servlet.Servlets;
import io.undertow.servlet.api.DeploymentInfo;
import io.undertow.servlet.api.DeploymentManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.DispatcherType;
import javax.servlet.ServletException;
import java.io.File;
import java.util.HashMap;
import java.util.Map;

/**
 * @author Feng Chen
 */
public class WebAppServer {

    private ServerConfiguration configuration;

    private ContextListener contextListener;

    private boolean started = false;

    private Undertow undertow;
    private DeploymentInfo deploymentInfo;
    private DeploymentManager deploymentManager;

    private Map<String, Object> properties = new HashMap<>();

    private static final String DIR_BIN = "bin";
    private static final String DIR_WEB = "webapp";
    private static final Logger logger = LoggerFactory.getLogger(WebAppServer.class);

    public WebAppServer(String contextPath, int port) {
        configuration = new ServerConfiguration();
        configuration.setContextPath(contextPath);
        configuration.setPort(port);
        AppServerCore.inst().setServerConfiguration(configuration);
        setupCoreInterceptors();
    }

    public WebAppServer(String contextPath, String ip, int port, int threads) {
        configuration = new ServerConfiguration();
        configuration.setContextPath(contextPath);
        configuration.setIp(ip);
        configuration.setPort(port);
        configuration.setIoThread(threads);
        configuration.setWorkThread(threads * 2);

        AppServerCore.inst().setServerConfiguration(configuration);
        setupCoreInterceptors();
    }

    public void setAdminUser(String username) {
        configuration.setAdminUser(username);
    }

    public void setAdminPwd(String password) {
        configuration.setAdminPwd(password);
    }

    public void setPackages(String... packages) {
        AppServerCore.inst().setPackages(packages);
    }

    public void setControllerFactory(ControllerFactory controllerFactory) {
        AppServerCore.inst().setControllerFactory(controllerFactory);
    }

    public void addInterceptor(Interceptor interceptor) {
        AppServerCore.inst().addInterceptor(interceptor);
    }

    public void addCoreInterceptor(Interceptor interceptor) {
        AppServerCore.inst().addCoreInterceptor(interceptor);
    }


    public void setContextListener(ContextListener contextListener) {
        this.contextListener = contextListener;
    }

    public void setProperty(String name, Object value) {
        properties.put(name, value);
    }

    public <T> T getProperty(String name) {
        return (T) properties.get(name);
    }

    public void start() {
        if (started) {
            logger.info("Already Started..");
            return;
        }

        Runtime.getRuntime().addShutdownHook(new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    stop();
                } catch (Exception e) {
                    logger.error("", e);
                }
            }
        }));

        try {
            configUndertow();
            undertow.start();

            if (contextListener != null) {
                contextListener.initialize(configuration);
            }

            started = true;
            logger.info("Web App Server Started Successfully. Server status : http://[server]:"
                    + configuration.getPort() + "/"
                    + configuration.getContextPath());

        } catch (ServletException e) {
            logger.error("", e);
            throw new AppServerException(e);
        }
    }

    public void stop() {
        if (!started) {
            return;
        }

        try {

            if (contextListener != null) {
                contextListener.destory(configuration);
            }

            deploymentManager.stop();
            undertow.stop();

            started = false;
            logger.info("Web App Server Closed Successfully. Good Bye.\n\n");
        } catch (ServletException e) {
            throw new AppServerException(e);
        }
    }

    // SECTION : INNER HELPER

    private void configUndertow() throws ServletException {
        deploymentInfo = Servlets.deployment();

        // 静态资源
        deploymentInfo.setClassLoader(WebAppServer.class.getClassLoader());
        deploymentInfo.setResourceManager(new CompositeResourceManager("src/main/webapp", "classpath:web", webroot()));
        deploymentInfo.setDeploymentName(configuration.getContextPath());
        deploymentInfo.setContextPath(configuration.getContextPath());
        deploymentInfo.setEagerFilterInit(true);

        // 添加核心过滤器
        deploymentInfo.addFilter(Servlets.filter("dispatch", DispatchFilter.class))
                .addFilterUrlMapping("dispatch", "/*", DispatcherType.REQUEST);
        deploymentManager = Servlets.defaultContainer().addDeployment(deploymentInfo);
        deploymentManager.deploy();

        HttpHandler httpHandler = deploymentManager.start();
        undertow = Undertow.builder()
                .setIoThreads(configuration.getIoThread())
                .setWorkerThreads(configuration.getWorkThread())
                .addHttpListener(configuration.getPort(), configuration.getIp())
                .setHandler(
                        Handlers.path().addPrefixPath(configuration.getContextPath(), httpHandler)
                )
                .build();
    }

    private void setupCoreInterceptors() {

        // 添加跨域拦截器
        addCoreInterceptor(new CorsInterceptor());
    }

    private static String webroot() {
        String userdir = System.getProperty("user.dir");
        if (userdir.contains(DIR_BIN)) {
            userdir = userdir.substring(0, userdir.indexOf(DIR_BIN));
        }
        return userdir + File.separator + DIR_WEB;
    }

}
