package com.ydn.web.appserver.core;

import io.undertow.server.handlers.resource.*;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

/**
 * 复合静态资源管理器
 *
 * @author Feng Chen
 */
public class CompositeResourceManager implements ResourceManager {

    private static final String CLASSPATH = "classpath:";

    private List<ResourceManager> list = new ArrayList<>();

    public CompositeResourceManager(String... paths) {
        for (String path : paths) {
            if (path.startsWith(CLASSPATH)) {
                list.add(newClassPathResourceManager(path.substring(CLASSPATH.length())));
            } else {
                list.add(newFileResourceManager(path));
            }
        }
    }

    @Override
    public Resource getResource(String path) throws IOException {
        for (ResourceManager rm : list) {
            Resource ret = rm.getResource(path);
            if (ret != null) {
                return ret;
            }
        }
        return null;
    }

    @Override
    public boolean isResourceChangeListenerSupported() {
        for (ResourceManager rm : list) {
            if (rm.isResourceChangeListenerSupported()) {
                return true;
            }
        }
        return false;
    }

    @Override
    public void registerResourceChangeListener(ResourceChangeListener listener) {
        for (ResourceManager rm : list) {
            rm.registerResourceChangeListener(listener);
        }
    }

    @Override
    public void removeResourceChangeListener(ResourceChangeListener listener) {
        for (ResourceManager rm : list) {
            if (rm.isResourceChangeListenerSupported()) {
                rm.removeResourceChangeListener(listener);
            }
        }
    }

    @Override
    public void close() throws IOException {
        for (ResourceManager rm : list) {
            rm.close();
        }
    }

    // SECTION : INNER HELPER

    /**
     * FileResourceManager 支持普通目录下加载 css、js 等 support 静态资源
     *
     * @param path
     * @return
     */
    private FileResourceManager newFileResourceManager(String path) {
        return new FileResourceManager(new File(path));
    }

    /**
     * ClassPathResourceManager 支持在 classpath 下与 jar 包内加载 css、js 等 support 静态资源
     *
     * @param path
     * @return
     */
    private ClassPathResourceManager newClassPathResourceManager(String path) {
        return new ClassPathResourceManager(CompositeResourceManager.class.getClassLoader(), path);
    }

}
