package com.ydn.web.appserver.core;

import com.ydn.web.appserver.AbstractController;
import com.ydn.web.appserver.annotation.Controller;
import com.ydn.web.appserver.annotation.RequestMapping;
import com.ydn.web.appserver.util.ClassUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author Feng Chen
 */
public class DefaultActionMappings implements ActionMappings {

    private Map<String, Action> mappings;
    private List<String> packages;

    private static final Logger logger = LoggerFactory.getLogger(DefaultActionMappings.class);

    public DefaultActionMappings() {
        packages = new ArrayList<>();
        mappings = new HashMap<>();
    }

    @Override
    public void setPackages(String... packages) {
        for (String aPackage : packages) {
            addPackage(aPackage);
        }
    }

    @Override
    public void addPackage(String aPackage) {
        if (!packages.contains(aPackage)) {
            packages.add(aPackage);
            lookup(aPackage);
        }
    }

    @Override
    public void addClass(Class<? extends AbstractController> aClass) {
        doLookup(aClass);
    }

    @Override
    public Action find(String route) {
        return mappings.get(route);
    }

    @Override
    public List<Action> list() {
        return new ArrayList<>(mappings.values());
    }

    // SECTION : INNER HELPER

    private void lookup(String aPackage) {
        try {
            List<Class<AbstractController>> clazzs = ClassUtils.findClasses(AbstractController.class, aPackage);
            for (Class<AbstractController> clazz : clazzs) {
                doLookup(clazz);
            }
        } catch (SecurityException e) {
            logger.error("", e);
        }
    }

    private void doLookup(Class<? extends AbstractController> clazz) {
        Controller res = null;
        // 查找前缀
        if (clazz.isAnnotationPresent(Controller.class)) {
            res = clazz.getAnnotation(Controller.class);
        }
        if (res != null) {
            // 遍历方法
            for (Method method : clazz.getDeclaredMethods()) {
                if (method.isAnnotationPresent(RequestMapping.class)) {
                    RequestMapping anno = method.getAnnotation(RequestMapping.class);
                    mappings.put(route(res.value(), anno.value()), new Action(clazz, method, res, anno));
                }
            }
        }
    }

    private String route(String prefix, String action) {
        StringBuilder builder = new StringBuilder();
        if (prefix != null) {
            if (!prefix.startsWith("/")) {
                builder.append("/");
            }
            builder.append(prefix);
        }
        if (!action.startsWith("/")) {
            builder.append("/");
        }
        builder.append(action);
        return builder.toString();
    }

}
