package com.ydn.web.appserver.core.dispatch;

import com.ydn.web.appserver.AbstractController;
import com.ydn.web.appserver.Result;
import com.ydn.web.appserver.annotation.Parameter;
import com.ydn.web.appserver.annotation.RequestMethod;
import com.ydn.web.appserver.core.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * @author Feng Chen
 */
public class ActionDispatcher implements Dispatcher {

    private static final Logger logger = LoggerFactory.getLogger(ActionDispatcher.class);

    @Override
    public void dispatch(HttpServletRequest request, HttpServletResponse response) throws Exception {

        // 设置上下文
        ActionContext actionContext = ActionContext.getContext();
        if (actionContext == null) {
            actionContext = new ActionContext();
            ActionContext.setContext(actionContext);
        }

        actionContext.setConfiguration(AppServerCore.inst().getServerConfiguration());
        actionContext.setRequest(request);
        actionContext.setResponse(response);

        String route = route(AppServerCore.inst().getServerConfiguration(), request);
        Action action = AppServerCore.inst()
                .getActionMappings()
                .find(route);
        if (action == null) {
            AppServerCore.inst()
                    .getRenderFactory()
                    .getErrorRender(404)
                    .setContext(request, response)
                    .render();
            return;
        }

        actionContext.setAction(action);

        // 校验请求方法
        if (!RequestMethod.ALL.name().equals(action.getAnno().method().name())) {
            if (!request.getMethod().equals(action.getAnno().method().name())) {
                AppServerCore.inst()
                        .getRenderFactory()
                        .getJsonRender(
                                Result.error(400, "请使用" + action.getAnno().method().name() + "方式")
                        )
                        .setContext(request, response)
                        .render();
                return;
            }
        }

        // 校验头部
        for (Parameter parameter : action.getAnno().header()) {
            if (parameter.required()) {
                if (actionContext.getHeader(parameter.name()) == null) {
                    AppServerCore.inst()
                            .getRenderFactory()
                            .getJsonRender(
                                    Result.error(400, parameter.name() + "必选")
                            )
                            .setContext(request, response)
                            .render();
                    return;
                }
            }
        }

        // 校验参数
        for (Parameter parameter : action.getAnno().request()) {
            if (parameter.required()) {
                if (actionContext.getParam(parameter.name()) == null) {
                    AppServerCore.inst()
                            .getRenderFactory()
                            .getJsonRender(
                                    Result.error(400, parameter.name() + "必选")
                            )
                            .setContext(request, response)
                            .render();
                    return;
                }
            }
        }

        doHandle();
    }


    // SECTION : INNER HELPER

    private String route(ServerConfiguration configuration, HttpServletRequest request) {
        String contextPath = configuration.getContextPath();
        int length = contextPath.length();
        if (!contextPath.startsWith("/")) {
            length++;
        }
        return request.getRequestURI().substring(length);
    }

    private void doHandle() {
        ActionContext actionContext = ActionContext.getContext();
        AbstractController controller = AppServerCore.inst().getControllerFactory().getController(actionContext.getAction().getClazz());
        if (controller != null) {
            execute(controller);
        } else {
            logger.error("没获取到 {} 实例", actionContext.getAction().getClazz().getName());
            AppServerCore.inst()
                    .getRenderFactory()
                    .getJsonRender(Result.error(500, "系统异常"))
                    .render();
        }
    }

    private void execute(AbstractController controller) {
        ActionContext actionContext = ActionContext.getContext();
        InterceptorChain interceptors = AppServerCore.inst()
                .getInterceptorFactory()
                .getInterceptors(
                        actionContext.getAction().getAnno().interceptor()
                );

        try {
            new ActionInvocation(interceptors, actionContext, new ActionInvocation.FinalInvocation() {
                @Override
                public void invoke() throws Exception {
                    actionContext.getAction().getMethod().invoke(controller);
                }
            }).invoke();
            addActionSuccStat(actionContext);
        } catch (Exception e) {
            logger.error("", e);
            addActionFailStat(actionContext, e);
            AppServerCore.inst()
                    .getRenderFactory()
                    .getJsonRender(Result.error(500, "系统异常"))
                    .render();
        }

    }

    private void addActionSuccStat(ActionContext actionContext) {
        AppServerCore.inst().getServerStat().succ(
                route(actionContext.getAction()),
                actionContext.getEndMillis() - actionContext.getStartMillis()
        );
    }

    private void addActionFailStat(ActionContext actionContext, Exception e) {
        AppServerCore.inst().getServerStat().fail(
                route(actionContext.getAction()),
                actionContext.getEndMillis() - actionContext.getStartMillis(),
                e
        );
    }

    private String route(Action action) {
        return new StringBuilder()
                .append(action.getRes().value())
                .append(action.getAnno().value())
                .toString();
    }


}
