package com.crazyxk.dta;

import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;

import com.crazyxk.dta.DtaModule;

import java.util.ArrayList;
import java.util.List;

/**
 * 用于解析 AndroidManifest 中的 Meta 属性，配合 {@link DtaModule} 使用
 *
 * @author Feng Chen
 */
public abstract class ManifestParser {

    private static final String MODULE_VALUE = "DtaModule";

    public static List<DtaModule> parse(Context context) {
        List<DtaModule> modules = new ArrayList<>();

        try {
            ApplicationInfo appInfo = context.getPackageManager().getApplicationInfo(
                    context.getPackageName(), PackageManager.GET_META_DATA
            );
            if (appInfo.metaData != null) {
                for (String key : appInfo.metaData.keySet()) {
                    if (MODULE_VALUE.equals(appInfo.metaData.get(key))) {
                        modules.add(parseModule(key));
                    }
                }
            }
        } catch (PackageManager.NameNotFoundException e) {
            throw new RuntimeException("Unable to find metadata to parse DtaModule", e);
        }

        return modules;
    }

    private static DtaModule parseModule(String className) {
        Class<?> clazz;
        try {
            clazz = Class.forName(className);
        } catch (ClassNotFoundException e) {
            throw new IllegalArgumentException("Unable to find DtaModule implementation", e);
        }

        Object module;
        try {
            module = clazz.newInstance();
        } catch (IllegalAccessException e) {
            throw new RuntimeException("Unable to instantiate DtaModule implementation for " + clazz, e);
        } catch (InstantiationException e) {
            throw new RuntimeException("Unable to instantiate DtaModule implementation for " + clazz, e);
        }

        if (!(module instanceof DtaModule)) {
            throw new RuntimeException("Expected instanceof DtaModule, but found: " + module);
        }

        return (DtaModule) module;
    }

}
