package com.crazyxk.dta;

import android.app.Application;
import android.content.Context;

import androidx.annotation.NonNull;

import com.crazyxk.dta.cache.Cache;
import com.crazyxk.dta.cache.Cache.Factory;
import com.crazyxk.dta.cache.CacheType;
import com.crazyxk.dta.util.Preconditions;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;

import javax.inject.Inject;
import javax.inject.Singleton;

import dagger.Lazy;
import io.reactivex.Observable;
import io.reactivex.Single;
import io.rx_cache2.internal.RxCache;
import retrofit2.Retrofit;

/**
 * @author Feng Chen
 */
@Singleton
public class RepositoryManager implements IRepositoryManager {

    @Inject
    Application mApplication;

    @Inject
    Lazy<Retrofit> mRetrofit;

    @Inject
    Lazy<RxCache> mRxCache;

    @Inject
    Factory mCacheFactory;

    private Cache<String, Object> mRetrofitServiceCache;
    private Cache<String, Object> mCacheServiceCache;

    @Inject
    public RepositoryManager() {

    }

    @NonNull
    @Override
    public <T> T obtainRetrofitService(@NonNull Class<T> serviceClass) {
        if (mRetrofitServiceCache == null) {
            mRetrofitServiceCache = mCacheFactory.build(CacheType.RETROFIT_SERVICE_CACHE);
        }
        Preconditions.checkNotNull(mRetrofitServiceCache,
                "Cannot return null from a Cache.Factory#build(int) method");
        T retrofitService = (T) mRetrofitServiceCache.get(serviceClass.getCanonicalName());
        if (retrofitService == null) {
            retrofitService = (T) Proxy.newProxyInstance(
                    serviceClass.getClassLoader(),
                    new Class[]{serviceClass},
                    new RetrofitServiceProxyHandler(mRetrofit.get(), serviceClass)
            );
            mRetrofitServiceCache.put(serviceClass.getCanonicalName(), retrofitService);
        }
        return retrofitService;
    }

    @Override
    public <T> T obtainCacheService(@NonNull Class<T> cacheClass) {
        Preconditions.checkNotNull(cacheClass, "cacheClass == null");
        if (mCacheServiceCache == null) {
            mCacheServiceCache = mCacheFactory.build(CacheType.CACHE_SERVICE_CACHE);
        }
        Preconditions.checkNotNull(mCacheServiceCache,
                "Cannot return null from a Cache.Factory#build(int) method");
        T cacheService = (T) mCacheServiceCache.get(cacheClass.getCanonicalName());
        if (cacheService == null) {
            cacheService = mRxCache.get().using(cacheClass);
            mCacheServiceCache.put(cacheClass.getCanonicalName(), cacheService);
        }
        return cacheService;
    }

    @Override
    public void clearAllCache() {
        mRxCache.get().evictAll().subscribe();
    }

    @NonNull
    @Override
    public Context getContext() {
        return mApplication;
    }

    ////////////////////////////////////////////////////////////////////

    private class RetrofitServiceProxyHandler implements InvocationHandler {

        private Retrofit mRetrofit;
        private Class<?> mServiceClass;
        private Object mRetrofitService;

        public RetrofitServiceProxyHandler(Retrofit retrofit, Class<?> serviceClass) {
            mRetrofit = retrofit;
            mServiceClass = serviceClass;
        }

        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
            if (method.getReturnType() == Observable.class) {
                return Observable.defer(() -> {
                    return (Observable) method.invoke(getRetrofitService(), args);
                });
            } else if (method.getReturnType() == Single.class) {
                return (Single) method.invoke(getRetrofitService(), args);
            }
            return method.invoke(getRetrofitService(), args);
        }

        // SECTION: INNER HELPER

        private Object getRetrofitService() {
            if (mRetrofitService == null) {
                mRetrofitService = mRetrofit.create(mServiceClass);
            }
            return mRetrofitService;
        }
    }

}
