package com.crazyxk.dta.base;

import android.app.Activity;
import android.app.Application;
import android.app.Application.ActivityLifecycleCallbacks;
import android.os.Bundle;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.FragmentActivity;
import androidx.fragment.app.FragmentManager.FragmentLifecycleCallbacks;

import com.crazyxk.dta.base.delegate.ActivityDelegate;
import com.crazyxk.dta.base.delegate.ActivityDelegateImpl;
import com.crazyxk.dta.base.delegate.IActivity;
import com.crazyxk.dta.cache.Cache;
import com.crazyxk.dta.cache.IntelligentCache;
import com.crazyxk.dta.DtaModule;

import java.util.List;

import javax.inject.Inject;
import javax.inject.Singleton;

import dagger.Lazy;

/**
 * @author Feng Chen
 */
@Singleton
public class ActivityLifecycle implements ActivityLifecycleCallbacks {

    @Inject
    AppManager mAppManager;

    @Inject
    Application mApplication;

    @Inject
    Cache<String, Object> mExtras;

    @Inject
    Lazy<FragmentLifecycleCallbacks> mFragmentLifecycle;

    @Inject
    Lazy<List<FragmentLifecycleCallbacks>> mFragmentLifecycles;

    @Inject
    public ActivityLifecycle() {

    }

    @Override
    public void onActivityCreated(@NonNull Activity activity, @Nullable Bundle savedInstanceState) {
        // 如果 intent 包含了此字段,并且为 true 说明不加入到 list 进行统一管理
        boolean isNotAdd = false;
        if (activity.getIntent() != null) {
            isNotAdd = activity.getIntent().getBooleanExtra(AppManager.IS_NOT_ADD_ACTIVITY_LIST, false);
        }

        if (!isNotAdd) {
            mAppManager.addActivity(activity);
        }

        // 配置 ActivityDelegate
        if (activity instanceof IActivity) {
            ActivityDelegate delegate = fetchActivityDelegate(activity);
            if (delegate == null) {
                Cache<String, Object> cache = getCacheFromActivity((IActivity) activity);
                delegate = new ActivityDelegateImpl(activity);
                cache.put(IntelligentCache.getKeyOfKeep(ActivityDelegate.ACTIVITY_DELEGATE), delegate);
            }
            delegate.onCreate(savedInstanceState);
        }

        registerFragmentCallbacks(activity);

    }

    @Override
    public void onActivityStarted(@NonNull Activity activity) {
        ActivityDelegate delegate = fetchActivityDelegate(activity);
        if (delegate != null) {
            delegate.onStart();
        }
    }

    @Override
    public void onActivityResumed(@NonNull Activity activity) {
        mAppManager.setCurrentActivity(activity);
        ActivityDelegate delegate = fetchActivityDelegate(activity);
        if (delegate != null) {
            delegate.onResume();
        }
    }

    @Override
    public void onActivityPaused(@NonNull Activity activity) {
        ActivityDelegate delegate = fetchActivityDelegate(activity);
        if (delegate != null) {
            delegate.onPause();
        }
    }

    @Override
    public void onActivityStopped(@NonNull Activity activity) {
        if (mAppManager.getCurrentActivity() == activity) {
            mAppManager.setCurrentActivity(null);
        }

        ActivityDelegate delegate = fetchActivityDelegate(activity);
        if (delegate != null) {
            delegate.onStop();
        }

    }

    @Override
    public void onActivitySaveInstanceState(@NonNull Activity activity, @NonNull Bundle outState) {
        ActivityDelegate delegate = fetchActivityDelegate(activity);
        if (delegate != null) {
            delegate.onSaveInstanceState(outState);
        }

    }

    @Override
    public void onActivityDestroyed(@NonNull Activity activity) {
        mAppManager.removeActivity(activity);
        ActivityDelegate delegate = fetchActivityDelegate(activity);
        if (delegate != null) {
            delegate.onDestroy();
            getCacheFromActivity((IActivity) activity).clear();
        }
    }


    // SECTION: INNER HELPER

    /**
     * 给每个 Activity 的所有 Fragment 设置监听其生命周期, Activity 可以通过 {@link IActivity#useFragment()}
     * 设置是否使用监听,如果这个 Activity 返回 false 的话,这个 Activity 下面的所有 Fragment 将不能使用 {@link FragmentDelegate}
     * 意味着 {@link DtaFragment} 也不能使用
     *
     * @param activity
     */
    private void registerFragmentCallbacks(Activity activity) {
        boolean useFragment = activity instanceof IActivity ? ((IActivity) activity).useFragment() : true;
        if (activity instanceof FragmentActivity && useFragment) {

            ((FragmentActivity) activity).getSupportFragmentManager()
                    .registerFragmentLifecycleCallbacks(
                            mFragmentLifecycle.get(), true
                    );

            if (mExtras.containsKey(IntelligentCache.getKeyOfKeep(DtaModule.class.getName()))) {
                List<DtaModule> configs = (List<DtaModule>) mExtras.get(
                        IntelligentCache.getKeyOfKeep(DtaModule.class.getName())
                );

                for (DtaModule config : configs) {
                    config.injectFragmentLifecycle(mApplication, mFragmentLifecycles.get());
                }

                mExtras.remove(IntelligentCache.getKeyOfKeep(DtaModule.class.getName()));
            }

            for (FragmentLifecycleCallbacks fragmentLifecycle : mFragmentLifecycles.get()) {
                ((FragmentActivity) activity).getSupportFragmentManager()
                        .registerFragmentLifecycleCallbacks(fragmentLifecycle, true);
            }
        }

    }

    private ActivityDelegate fetchActivityDelegate(Activity activity) {
        ActivityDelegate delegate = null;
        if (activity instanceof IActivity) {
            Cache<String, Object> cache = getCacheFromActivity((IActivity) activity);
            delegate = (ActivityDelegate) cache.get(
                    IntelligentCache.getKeyOfKeep(ActivityDelegate.ACTIVITY_DELEGATE)
            );
        }
        return delegate;
    }

    @NonNull
    public Cache<String, Object> getCacheFromActivity(IActivity activity) {
        return activity.provideCache();
    }


}
