package com.crazyxk.dta.base;

import android.app.Activity;
import android.content.Context;
import android.os.Bundle;
import android.util.AttributeSet;
import android.view.InflateException;
import android.view.View;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.app.AppCompatActivity;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentManager;

import com.crazyxk.dta.base.delegate.IActivity;
import com.crazyxk.dta.cache.Cache;
import com.crazyxk.dta.cache.CacheType;
import com.crazyxk.dta.rxlifecycle.ActivityLifecycleable;
import com.crazyxk.dta.mvp.DtaPresenter;
import com.crazyxk.dta.util.DtaUtil;
import com.trello.rxlifecycle2.android.ActivityEvent;

import javax.inject.Inject;

import butterknife.ButterKnife;
import butterknife.Unbinder;
import io.reactivex.subjects.BehaviorSubject;
import io.reactivex.subjects.Subject;

import static com.crazyxk.dta.util.ThirdViewUtil.convertAutoView;

/**
 * @author Feng Chen
 */
public abstract class DtaActivity<P extends DtaPresenter>
        extends AppCompatActivity implements IActivity, ActivityLifecycleable {

    private Unbinder mUnbinder;
    private Cache<String, Object> mCache;

    private final BehaviorSubject<ActivityEvent> mLifecycleSubject = BehaviorSubject.create();

    @Inject
    @Nullable
    protected P mPresenter;

    @NonNull
    @Override
    public Cache<String, Object> provideCache() {
        if (mCache == null) {
            mCache = DtaUtil.obtainAppComponentFromContext(this)
                    .cacheFactory()
                    .build(CacheType.ACTIVITY_CACHE);
        }
        return mCache;
    }

    @NonNull
    @Override
    public Subject<ActivityEvent> provideLifecycleSubject() {
        return mLifecycleSubject;
    }

    @Override
    public View onCreateView(String name, Context context, AttributeSet attrs) {
        View view = convertAutoView(name, context, attrs);
        return view == null ? super.onCreateView(name, context, attrs) : view;
    }

    @Override
    protected void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        try {
            int layout = initView(savedInstanceState);
            // 如果initView返回0,框架则不会调用setContentView(),当然也不会 Bind ButterKnife
            if (layout != 0) {
                setContentView(layout);

                // 绑定 butterknife
                mUnbinder = ButterKnife.bind(this);
            }
        } catch (Exception e) {
            if (e instanceof InflateException) throw e;
            e.printStackTrace();
        }

        initData(savedInstanceState);
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        if (mUnbinder != null && mUnbinder != Unbinder.EMPTY) {
            mUnbinder.unbind();
            mUnbinder = null;
        }

        if (mPresenter != null) {
            mPresenter.onDestroy(); // 释放资源
            mPresenter = null;
        }
    }

    /**
     * 是否使用 EventBus
     *
     * @return 返回 {@code true} (默认为 {@code true}), KITS 会自动注册 EventBus
     */
    @Override
    public boolean useEventBus() {
        return true;
    }

    /**
     * 这个 {@link Activity} 是否会使用 {@link Fragment}, 框架会根据这个属性判断是否注册 {@link FragmentManager.FragmentLifecycleCallbacks}
     * *如果返回 {@code false}, 那意味着这个 {@link Activity} 不需要绑定 {@link Fragment}, 那你再在这个 {@link Activity} 中绑定继承于 {@link DtaFragment} 的 {@link Fragment} 将不起任何作用
     *
     * @return 返回 {@code true} (默认为 {@code true}), 则需要使用 {@link Fragment}
     */
    @Override
    public boolean useFragment() {
        return true;
    }
}
