package com.crazyxk.dta.base;

import android.content.Context;
import android.os.Bundle;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.Fragment;

import com.crazyxk.dta.base.delegate.IFragment;
import com.crazyxk.dta.cache.Cache;
import com.crazyxk.dta.cache.CacheType;
import com.crazyxk.dta.rxlifecycle.FragmentLifecycleable;
import com.crazyxk.dta.mvp.DtaPresenter;
import com.crazyxk.dta.util.DtaUtil;
import com.trello.rxlifecycle2.android.FragmentEvent;

import javax.inject.Inject;

import butterknife.ButterKnife;
import butterknife.Unbinder;
import io.reactivex.subjects.BehaviorSubject;
import io.reactivex.subjects.Subject;

/**
 * @author Feng Chen
 */
public abstract class DtaFragment<P extends DtaPresenter>
        extends Fragment implements IFragment, FragmentLifecycleable {

    private Context mContext;
    private Unbinder mUnbinder;
    private Cache<String, Object> mCache;
    private final BehaviorSubject<FragmentEvent> mLifecycleSubject = BehaviorSubject.create();

    @Inject
    @Nullable
    protected P mPresenter;  //如果当前页面逻辑简单, Presenter 可以为 null

    @NonNull
    @Override
    public Cache<String, Object> provideCache() {
        if (mCache == null) {
            mCache = DtaUtil.obtainAppComponentFromContext(getActivity())
                    .cacheFactory()
                    .build(CacheType.FRAGMENT_CACHE);
        }
        return mCache;
    }

    @NonNull
    @Override
    public Subject<FragmentEvent> provideLifecycleSubject() {
        return mLifecycleSubject;
    }

    @Override
    public void onAttach(Context context) {
        super.onAttach(context);
        mContext = context;
    }

    @Nullable
    @Override
    public View onCreateView(@NonNull LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {
        View view = initView(inflater, container, savedInstanceState);
        mUnbinder = ButterKnife.bind(this, view);
        return view;
    }

    @Override
    public void onActivityCreated(@Nullable Bundle savedInstanceState) {
        super.onActivityCreated(savedInstanceState);
        initData(savedInstanceState);
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        if (mUnbinder != null && mUnbinder != Unbinder.EMPTY) {
            mUnbinder.unbind();
            mUnbinder = null;
        }

        if (mPresenter != null) {
            mPresenter.onDestroy();
            mPresenter = null;
        }
    }

    @Override
    public void onDetach() {
        super.onDetach();
        mContext = null;
    }

    /**
     * 是否使用 EventBus
     *
     * @return 返回 {@code true} (默认为 {@code true}), KITS 会自动注册 EventBus
     */
    @Override
    public boolean useEventBus() {
        return true;
    }
}

