package com.crazyxk.dta.base.delegate;

import android.app.Activity;
import android.app.Application;
import android.app.Application.ActivityLifecycleCallbacks;
import android.app.Service;
import android.content.ComponentCallbacks2;
import android.content.ContentProvider;
import android.content.Context;
import android.content.res.Configuration;

import androidx.annotation.NonNull;
import androidx.fragment.app.Fragment;

import com.crazyxk.dta.base.App;
import com.crazyxk.dta.di.component.AppComponent;
import com.crazyxk.dta.di.component.DaggerAppComponent;
import com.crazyxk.dta.di.module.ConfigModule;
import com.crazyxk.dta.DtaModule;
import com.crazyxk.dta.ManifestParser;
import com.crazyxk.dta.cache.IntelligentCache;

import java.util.ArrayList;
import java.util.List;

import javax.inject.Inject;
import javax.inject.Named;

/**
 * AppDelegate 代理 Application 的生命周期，在对应的生命周期，执行对应的逻辑
 *
 * @author Feng Chen
 */
public class AppDelegate implements App, AppLifecycle {

    private Application mApplication;
    private AppComponent mAppComponent;

    private List<DtaModule> mModules;
    private List<AppLifecycle> mAppLifecycles = new ArrayList<>();
    private List<ActivityLifecycleCallbacks> mActivityLifecycles = new ArrayList<>();
    private ComponentCallbacks2 mComponentCallback;

    @Inject
    @Named("ActivityLifecycle")
    protected ActivityLifecycleCallbacks mActivityLifecycle;

    @Inject
    @Named("RxActivityLifecycle")
    protected ActivityLifecycleCallbacks mRxActivityLifecycle;


    public AppDelegate(@NonNull Context context) {
        // 通过反射， 将 AndroidManifest.xml 中带有 DtaModule 标签的 class 转成对象集合 (List<DtaModule>)
        mModules = ManifestParser.parse(context);

        // 遍历之前获得的集合, 执行每一个 ConfigModule 实现类的某些方法
        for (DtaModule module : mModules) {

            //将框架外部, 开发者实现的 Application 的生命周期回调 (AppLifecycles) 存入 mAppLifecycles 集合 (此时还未注册回调)
            module.injectAppLifecycle(context, mAppLifecycles);

            //将框架外部, 开发者实现的 Activity 的生命周期回调 (ActivityLifecycleCallbacks) 存入 mActivityLifecycles 集合 (此时还未注册回调)
            module.injectActivityLifecycle(context, mActivityLifecycles);
        }

    }

    @NonNull
    @Override
    public AppComponent getAppComponent() {
        return mAppComponent;
    }

    @Override
    public void attachBaseContext(@NonNull Context context) {
        for (AppLifecycle lifecycle : mAppLifecycles) {
            lifecycle.attachBaseContext(context);
        }
    }

    @Override
    public void onCreate(@NonNull Application application) {
        mApplication = application;
        mAppComponent = DaggerAppComponent
                .builder()
                .application(mApplication)
                .configModule(getConfigModule(mApplication, mModules)) // 全局配置
                .build();

        mAppComponent.inject(this);

        // 将 DtaModule 的实现类的集合存放到缓存 Cache, 可以随时获取
        mAppComponent.extras().put(
                IntelligentCache.getKeyOfKeep(DtaModule.class.getName()), mModules
        );
        mModules = null;

        // 注册框架内部实现的 Activity 生命周期逻辑
        mApplication.registerActivityLifecycleCallbacks(mActivityLifecycle);

        // 注册框架内部实现的 RxActivity 逻辑
        mApplication.registerActivityLifecycleCallbacks(mRxActivityLifecycle);

        // 注册框架外部， 开发者扩展的  Activity 生命周期逻辑
        // 每个 DtaModule 的实现类可以声明多个 Activity 的生命周期回调
        // 也可以有 N 个 ConfigModule 的实现类 (完美支持组件化项目 各个 Module 的各种独特需求)
        for (ActivityLifecycleCallbacks lifecycle : mActivityLifecycles) {
            mApplication.registerActivityLifecycleCallbacks(lifecycle);
        }

        mComponentCallback = new AppComponentCallbacks(mApplication, mAppComponent);

        // 注册回调： 内存紧张时释放部分内存
        mApplication.registerComponentCallbacks(mComponentCallback);

        // 执行框架外部， 开发者扩展的 App onCreate 逻辑
        for (AppLifecycle lifecycle : mAppLifecycles) {
            lifecycle.onCreate(mApplication);
        }

    }

    @Override
    public void onTerminate(@NonNull Application application) {
        if (mActivityLifecycle != null) {
            mApplication.unregisterActivityLifecycleCallbacks(mActivityLifecycle);
        }

        if (mRxActivityLifecycle != null) {
            mApplication.unregisterActivityLifecycleCallbacks(mRxActivityLifecycle);
        }

        if (mComponentCallback != null) {
            mApplication.unregisterComponentCallbacks(mComponentCallback);
        }

        if (mActivityLifecycles != null && mActivityLifecycles.size() > 0) {
            for (ActivityLifecycleCallbacks lifecycle : mActivityLifecycles) {
                mApplication.unregisterActivityLifecycleCallbacks(lifecycle);
            }
        }

        mAppComponent = null;
        mActivityLifecycle = null;
        mRxActivityLifecycle = null;
        mActivityLifecycles = null;
        mComponentCallback = null;
        mAppLifecycles = null;
        mApplication = null;
    }

    // SECTION : INNER HELPER

    private ConfigModule getConfigModule(Context context, List<DtaModule> modules) {
        ConfigModule.Builder builder = ConfigModule.builder();
        //  遍历 DtaModule 集合, 给配置 ConfigModule 添加参数
        for (DtaModule module : modules) {
            module.applyOptions(context, builder);
        }
        return builder.build();
    }

    ///////////////////////////////////////////////////////

    private static class AppComponentCallbacks implements ComponentCallbacks2 {

        private Application mApplication;
        private AppComponent mAppComponent;

        AppComponentCallbacks(Application application, AppComponent appComponent) {
            this.mApplication = application;
            this.mAppComponent = appComponent;
        }

        /**
         * 在你的 App 生命周期的任何阶段, {@link ComponentCallbacks2#onTrimMemory(int)} 发生的回调都预示着你设备的内存资源已经开始紧张
         * 你应该根据 {@link ComponentCallbacks2#onTrimMemory(int)} 发生回调时的内存级别来进一步决定释放哪些资源
         * {@link ComponentCallbacks2#onTrimMemory(int)} 的回调可以发生在 {@link Application}、{@link Activity}、{@link Service}、{@link ContentProvider}、{@link Fragment}
         *
         * @param level 内存级别
         * @see <a href="https://developer.android.com/reference/android/content/ComponentCallbacks2.html#TRIM_MEMORY_RUNNING_MODERATE">level 官方文档</a>
         */
        @Override
        public void onTrimMemory(int level) {
            //状态1. 当开发者的 App 正在运行
            //设备开始运行缓慢, 不会被 kill, 也不会被列为可杀死的, 但是设备此时正运行于低内存状态下, 系统开始触发杀死 LRU 列表中的进程的机制
//                case TRIM_MEMORY_RUNNING_MODERATE:


            //设备运行更缓慢了, 不会被 kill, 但请你回收 unused 资源, 以便提升系统的性能, 你应该释放不用的资源用来提升系统性能 (但是这也会直接影响到你的 App 的性能)
//                case TRIM_MEMORY_RUNNING_LOW:


            //设备运行特别慢, 当前 App 还不会被杀死, 但是系统已经把 LRU 列表中的大多数进程都已经杀死, 因此你应该立即释放所有非必须的资源
            //如果系统不能回收到足够的 RAM 数量, 系统将会清除所有的 LRU 列表中的进程, 并且开始杀死那些之前被认为不应该杀死的进程, 例如那个包含了一个运行态 Service 的进程
//                case TRIM_MEMORY_RUNNING_CRITICAL:


            //状态2. 当前 App UI 不再可见, 这是一个回收大个资源的好时机
//                case TRIM_MEMORY_UI_HIDDEN:


            //状态3. 当前的 App 进程被置于 Background LRU 列表中
            //进程位于 LRU 列表的上端, 尽管你的 App 进程并不是处于被杀掉的高危险状态, 但系统可能已经开始杀掉 LRU 列表中的其他进程了
            //你应该释放那些容易恢复的资源, 以便于你的进程可以保留下来, 这样当用户回退到你的 App 的时候才能够迅速恢复
//                case TRIM_MEMORY_BACKGROUND:


            //系统正运行于低内存状态并且你的进程已经已经接近 LRU 列表的中部位置, 如果系统的内存开始变得更加紧张, 你的进程是有可能被杀死的
//                case TRIM_MEMORY_MODERATE:


            //系统正运行于低内存的状态并且你的进程正处于 LRU 列表中最容易被杀掉的位置, 你应该释放任何不影响你的 App 恢复状态的资源
            //低于 API 14 的 App 可以使用 onLowMemory 回调
//                case TRIM_MEMORY_COMPLETE:
        }

        @Override
        public void onConfigurationChanged(Configuration newConfig) {

        }

        /**
         * 当系统开始清除 LRU 列表中的进程时, 尽管它会首先按照 LRU 的顺序来清除, 但是它同样会考虑进程的内存使用量, 因此消耗越少的进程则越容易被留下来
         * {@link ComponentCallbacks2#onTrimMemory(int)} 的回调是在 API 14 才被加进来的, 对于老的版本, 你可以使用 {@link ComponentCallbacks2#onLowMemory} 方法来进行兼容
         * {@link ComponentCallbacks2#onLowMemory} 相当于 {@code onTrimMemory(TRIM_MEMORY_COMPLETE)}
         *
         * @see #TRIM_MEMORY_COMPLETE
         */
        @Override
        public void onLowMemory() {
            //系统正运行于低内存的状态并且你的进程正处于 LRU 列表中最容易被杀掉的位置, 你应该释放任何不影响你的 App 恢复状态的资源
        }
    }

}
