package com.crazyxk.dta.di.component;

import android.app.Application;
import android.content.Context;

import com.crazyxk.dta.base.delegate.AppDelegate;
import com.crazyxk.dta.di.module.AppModule;
import com.crazyxk.dta.di.module.ClientModule;
import com.crazyxk.dta.di.module.ConfigModule;
import com.crazyxk.dta.base.AppManager;
import com.crazyxk.dta.DtaModule;
import com.crazyxk.dta.IRepositoryManager;
import com.crazyxk.dta.cache.Cache;
import com.crazyxk.dta.imageloader.ImageLoader;
import com.google.gson.Gson;

import java.io.File;
import java.util.concurrent.ExecutorService;

import javax.inject.Singleton;

import dagger.BindsInstance;
import dagger.Component;
import me.jessyan.rxerrorhandler.core.RxErrorHandler;
import okhttp3.OkHttpClient;

/**
 * @author Feng Chen
 */
@Singleton
@Component(modules = {AppModule.class, ClientModule.class, ConfigModule.class})
public interface AppComponent {

    Application application();

    AppManager appManager();

    /**
     * 用于管理网络请求层, 以及数据缓存层
     *
     * @return
     */
    IRepositoryManager repositoryManager();

    /**
     * RxJava 错误处理管理类
     *
     * @return
     */
    RxErrorHandler rxErrorHandler();

    /**
     * 图片加载管理器, 用于加载图片的管理类, 使用策略者模式, 可在运行时动态替换任何图片加载框架
     *
     * @return
     */
    ImageLoader imageLoader();

    /**
     * 网络请求框架
     *
     * @return {@link OkHttpClient}
     */
    OkHttpClient okHttpClient();

    /**
     * Json 序列化库
     *
     * @return {@link Gson}
     */
    Gson gson();

    /**
     * 缓存文件根目录 (RxCache 和 Glide 的缓存都已经作为子文件夹放在这个根目录下), 应该将所有缓存都统一放到这个根目录下
     * 便于管理和清理, 可在 {@link DtaModule#applyOptions(Context, ConfigModule.Builder)} 种配置
     *
     * @return {@link File}
     */
    File cacheFile();

    /**
     * 用来存取一些整个 App 公用的数据, 切勿大量存放大容量数据, 这里的存放的数据和 {@link Application} 的生命周期一致
     *
     * @return {@link Cache}
     */
    Cache<String, Object> extras();

    /**
     * 用于创建框架所需缓存对象的工厂
     *
     * @return {@link Cache.Factory}
     */
    Cache.Factory cacheFactory();

    /**
     * 返回一个全局公用的线程池,适用于大多数异步需求。
     * 避免多个线程池创建带来的资源消耗。
     *
     * @return {@link ExecutorService}
     */
    ExecutorService executorService();

    void inject(AppDelegate delegate);

    ///////////////////////////////////////////////////////////////////

    @Component.Builder
    interface Builder {

        @BindsInstance
        Builder application(Application application);

        Builder configModule(ConfigModule configModule);

        AppComponent build();
    }


}
