package com.crazyxk.dta.di.module;

import android.app.Application;
import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.crazyxk.dta.okhttp.GlobalHttpHandler;
import com.crazyxk.dta.okhttp.log.RequestInterceptor;
import com.crazyxk.dta.util.DataHelper;
import com.google.gson.Gson;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

import javax.inject.Named;
import javax.inject.Singleton;

import dagger.Binds;
import dagger.Module;
import dagger.Provides;
import io.rx_cache2.internal.RxCache;
import io.victoralbertos.jolyglot.GsonSpeaker;
import me.jessyan.rxerrorhandler.core.RxErrorHandler;
import me.jessyan.rxerrorhandler.handler.listener.ResponseErrorListener;
import okhttp3.Dispatcher;
import okhttp3.HttpUrl;
import okhttp3.Interceptor;
import okhttp3.OkHttpClient;
import okhttp3.Response;
import retrofit2.Retrofit;
import retrofit2.adapter.rxjava2.RxJava2CallAdapterFactory;
import retrofit2.converter.gson.GsonConverterFactory;

/**
 * DTA 提供第三方客户端实例
 *
 * @author Feng Chen
 */
@Module
public abstract class ClientModule {

    private static final int TIMEOUT = 10;

    /**
     * 提供 {@link Retrofit}
     *
     * @param application
     * @param configuration
     * @param builder
     * @param client
     * @param httpUrl
     * @param gson
     * @return
     */
    @Singleton
    @Provides
    static Retrofit provideRetrofit(
            Application application, @Nullable RetrofitConfiguration configuration,
            Retrofit.Builder builder, OkHttpClient client, HttpUrl httpUrl, Gson gson
    ) {
        builder.baseUrl(httpUrl)
                .client(client);

        if (configuration != null) {
            configuration.configRetrofit(application, builder);
        }

        builder.addCallAdapterFactory(RxJava2CallAdapterFactory.create()) // 使用 RxJava
                .addConverterFactory(GsonConverterFactory.create()); // 使用 Gson

        return builder.build();
    }

    @Singleton
    @Provides
    static Retrofit.Builder provideRetrofitBuilder() {
        return new Retrofit.Builder();
    }

    /**
     * 提供 {@link OkHttpClient}
     *
     * @param application
     * @param configuration
     * @param builder
     * @param interceptor
     * @param interceptors
     * @param handler
     * @param executorService
     * @return
     */
    @Singleton
    @Provides
    static OkHttpClient provideOkHttpClient(
            Application application, @Nullable OkHttpConfiguration configuration,
            OkHttpClient.Builder builder, Interceptor interceptor, @Nullable List<Interceptor> interceptors,
            @Nullable GlobalHttpHandler handler, ExecutorService executorService
    ) {
        builder.connectTimeout(TIMEOUT, TimeUnit.SECONDS)
                .readTimeout(TIMEOUT, TimeUnit.SECONDS)
                .writeTimeout(TIMEOUT, TimeUnit.SECONDS)
                .addNetworkInterceptor(interceptor);

        if (handler != null) {
            builder.addInterceptor(new Interceptor() {
                @Override
                public Response intercept(Chain chain) throws IOException {
                    return chain.proceed(handler.onHttpRequestBefore(chain, chain.request()));
                }
            });
        }

        if (interceptors != null) {
            for (Interceptor inter : interceptors) {
                builder.addInterceptor(inter);
            }
        }

        // 设置线程池
        builder.dispatcher(new Dispatcher(executorService));

        if (configuration != null) {
            configuration.configOkHttp(application, builder);
        }

        return builder.build();
    }

    @Singleton
    @Provides
    static OkHttpClient.Builder provideOkHttpClientBuilder() {
        return new OkHttpClient.Builder();
    }

    @Binds
    abstract Interceptor bindInterceptor(RequestInterceptor interceptor);

    /**
     * 提供 {@link RxCache}
     *
     * @param application
     * @param configuration
     * @param builder
     * @param cacheDirectory
     * @param gson
     * @return
     */
    @Singleton
    @Provides
    static RxCache provideRxCache(
            Application application, @Nullable RxCacheConfiguration configuration,
            RxCache.Builder builder, @Named("RxCacheDirectory") File cacheDirectory,
            Gson gson
    ) {
        if (configuration != null) {
            configuration.configRxCache(application, builder);
        }

        return builder.persistence(cacheDirectory, new GsonSpeaker(gson));
    }

    @Singleton
    @Provides
    static RxCache.Builder provideRxCacheBuilder() {
        return new RxCache.Builder();
    }

    @Singleton
    @Provides
    @Named("RxCacheDirectory")
    static File provideRxCacheDirectory(File cacheDir) {
        return DataHelper.makeDirs(new File(cacheDir, "RxCache"));
    }

    /**
     * 提供处理 RxJava 错误管理器
     *
     * @param application
     * @param listener
     * @return
     */
    @Singleton
    @Provides
    static RxErrorHandler provideRxErrorHandler(Application application, ResponseErrorListener listener) {
        return RxErrorHandler
                .builder()
                .with(application)
                .responseErrorListener(listener)
                .build();
    }

    /////////////////////////////////////////////////////////////////

    public interface RetrofitConfiguration {

        void configRetrofit(@NonNull Context context, @NonNull Retrofit.Builder builder);

    }

    public interface OkHttpConfiguration {

        void configOkHttp(@NonNull Context context, @NonNull OkHttpClient.Builder builder);

    }

    public interface RxCacheConfiguration {

        void configRxCache(@NonNull Context context, @NonNull RxCache.Builder builder);

    }


}
