package com.crazyxk.dta.mvp;

import android.app.Activity;

import androidx.lifecycle.Lifecycle;
import androidx.lifecycle.LifecycleObserver;
import androidx.lifecycle.LifecycleOwner;
import androidx.lifecycle.OnLifecycleEvent;

import com.crazyxk.dta.eventbus.EventBusManager;

/**
 * Presenter 基类
 *
 * @author Feng Chen
 */
public class BaseDtaPresenter<M extends DtaModel, V extends DtaView> implements DtaPresenter, LifecycleObserver {

    protected M mModel;
    protected V mView;

    public BaseDtaPresenter(M model, V view) {
        mModel = model;
        mView = view;
        onStart();
    }

    public BaseDtaPresenter(V view) {
        mView = view;
        onStart();
    }

    public BaseDtaPresenter() {
        onStart();
    }

    @Override
    public void onStart() {
        //将 LifecycleObserver 注册给 LifecycleOwner 后 @OnLifecycleEvent 才可以正常使用
        if (mView != null && mView instanceof LifecycleObserver) {
            ((LifecycleOwner) mView).getLifecycle().addObserver(this);

            if (mModel != null && mModel instanceof LifecycleObserver) {
                ((LifecycleOwner) mView).getLifecycle().addObserver((LifecycleObserver) mModel);
            }
        }

        if (useEventBus()) {
            EventBusManager.getInstance().register(this);
        }

    }

    /**
     * 在框架中 {@link Activity#onDestroy() } 时会默认调用 {@link DtaPresenter#onDestroy()}
     */
    @Override
    public void onDestroy() {
        if (useEventBus()) {
            EventBusManager.getInstance().unregister(this); // 注销 EventBus
        }

        if (mModel != null) {
            mModel.onDestroy();
        }

        mModel = null;
        mView = null;
    }

    @OnLifecycleEvent(Lifecycle.Event.ON_DESTROY)
    void onDestroy(LifecycleOwner owner) {
        owner.getLifecycle().removeObserver(this);
    }

    /**
     * 是否使用 EventBus
     * DTA 支持 EventBus 和 GreenRobot
     *
     * @return
     */
    public boolean useEventBus() {
        return true;
    }


}
