package com.ydn.simplelock.adapter;

import com.ydn.simplelock.configuration.LockConfiguration;
import com.ydn.simplelock.configuration.RedisLockConfiguration;
import com.ydn.simplelock.configuration.ZkLockConfiguration;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * @author Feng Chen
 */
public class LockAdapterFactory {

    private final static Map<Class<? extends LockConfiguration>, LockAdapterCreator> CREATOR_MAP
            = new HashMap<Class<? extends LockConfiguration>, LockAdapterCreator>();

    static {
        CREATOR_MAP.put(ZkLockConfiguration.class, new ZkLockAdapterCreator());
        CREATOR_MAP.put(RedisLockConfiguration.class, new RedisLockAdapterCreator());
    }

    public static LockAdapter createLockAdapter(LockConfiguration configuration) {
        Class cacheConfigType;
        boolean isCompatibleType;
        Iterator it = CREATOR_MAP.keySet().iterator();
        do {
            if (!it.hasNext()) {
                throw new IllegalArgumentException("Unrecognized configuration type: " + configuration.getClass());
            }

            cacheConfigType = (Class) it.next();
            isCompatibleType = cacheConfigType.isAssignableFrom(configuration.getClass());
        } while (!isCompatibleType);

        return CREATOR_MAP.get(configuration.getClass()).create(configuration);
    }


    private static class ZkLockAdapterCreator implements LockAdapterCreator<ZkLockConfiguration> {
        @Override
        public LockAdapter create(ZkLockConfiguration configuration) {
            return new ZkLockAdapter(configuration);
        }
    }

    private static class RedisLockAdapterCreator implements LockAdapterCreator<RedisLockConfiguration> {
        @Override
        public LockAdapter create(RedisLockConfiguration configuration) {
            return new RedisLockAdapter(configuration);
        }
    }


    ////////////////////////////////////

    private static interface LockAdapterCreator<T extends LockConfiguration> {
        LockAdapter create(T configuration);
    }

}
